#include "ptexlib.h"
#include "image.h"
#include <kpathsea/c-memstr.h>

image_entry *image_ptr, *image_tab = 0;
integer image_max;

int epdf_width;
int epdf_height;
int epdf_orig_x;
int epdf_orig_y;
int epdf_selected_page;
void *epdf_doc;
void *epdf_xref;

static integer new_image_entry(void)
{
    image_entry *p;
    if (image_tab != 0) {
        for (p = image_tab; p < image_ptr; p++)
            if (p->image_type == IMAGE_TYPE_NONE) /* found an unused entry */
                break;
    }
    if (image_tab == 0 || p == image_ptr) {
        entry_room(image, 256);
        p = image_ptr++;
    }
    p->image_type = 0;
    p->x_res = 0;
    p->y_res = 0;
    p->width = 0;
    p->height = 0;
    return p - image_tab;
}

integer imagecolor(integer img)
{
    return img_color(img);
}

integer imagewidth(integer img)
{
    return img_ptr(img)->width;
}

integer imageheight(integer img)
{
    return img_ptr(img)->height;
}

integer imagexres(integer img)
{
    return img_ptr(img)->x_res;
}

integer imageyres(integer img)
{
    return img_ptr(img)->y_res;
}

boolean ispdfimage(integer img)
{
    return img_type(img) == IMAGE_TYPE_PDF;
}

boolean checkimageb(integer procset)
{
    return procset & IMAGE_COLOR_B;
}

boolean checkimagec(integer procset)
{
    return procset & IMAGE_COLOR_C;
}

boolean checkimagei(integer procset)
{
    return procset & IMAGE_COLOR_I;
}

void updateimageprocset(integer procset)
{
    pdfimageprocset |= procset;
}

integer epdforigx(integer img)
{
    return pdf_ptr(img)->orig_x;
}

integer epdforigy(integer img)
{
    return pdf_ptr(img)->orig_y;
}

integer readimage(strnumber s, integer page_num)
{
    char *image_suffix;
    integer img = new_image_entry();
    cur_file_name = makecstring(s);
    img_name(img) = kpse_find_file(cur_file_name, kpse_tex_format, true);
    if (img_name(img) == 0)
        pdftex_fail("cannot find image file");
    if ((image_suffix = rindex(cur_file_name, '.')) == 0)
        pdftex_fail("cannot find image file name extension");
    if (strcasecmp(image_suffix, ".pdf") == 0) {
        img_type(img) = IMAGE_TYPE_PDF;
        pdf_ptr(img) = xtalloc(1, pdf_image_struct);
        read_pdf_info(img_name(img), page_num);
        img_ptr(img)->width = epdf_width;
        img_ptr(img)->height = epdf_height;
        pdf_ptr(img)->orig_x = epdf_orig_x;
        pdf_ptr(img)->orig_y = epdf_orig_y;
        pdf_ptr(img)->selected_page = page_num;
        pdf_ptr(img)->xref = epdf_xref;
        pdf_ptr(img)->doc = epdf_doc;
    }
    else if (strcasecmp(image_suffix, ".png") == 0) {
        img_type(img) = IMAGE_TYPE_PNG;
        read_png_info(img);
    }
    else if (strcasecmp(image_suffix, ".jpg") == 0 ||
             strcasecmp(image_suffix, ".jpeg") == 0) {
        jpg_ptr(img) = xtalloc(1, JPG_IMAGE_INFO);
        img_type(img) = IMAGE_TYPE_JPG;
        read_jpg_info(img);
    }
#ifdef HAVE_TIFF
    else if (strcasecmp(image_suffix, ".tif") == 0 ||
             strcasecmp(image_suffix, ".tiff") == 0) {
        img_type(img) = IMAGE_TYPE_TIF;
        read_tiff_info(img);
    }
#endif
    else 
        pdftex_fail("unknown type of image");
    cur_file_name = 0;
    return img;
}

void writeimage(integer img)
{
    tex_printf(" <%s", img_name(img));
    switch (img_type(img)) {
    case IMAGE_TYPE_PNG:
        write_png(img);
        break;
    case IMAGE_TYPE_JPG:
        write_jpg(img);
        break;
#ifdef HAVE_TIFF
    case IMAGE_TYPE_TIF:
        write_tiff(img);
        break;
#endif
    case IMAGE_TYPE_PDF:
        epdf_xref = pdf_ptr(img)->xref;
        epdf_doc = pdf_ptr(img)->doc;
        epdf_selected_page = pdf_ptr(img)->selected_page;
        write_epdf();
        break;
    default:
        pdftex_fail("unknown type of image");
    }
    tex_printf(">");
}

void deleteimage(integer img)
{
    switch (img_type(img)) {
    case IMAGE_TYPE_PDF:
        epdf_xref = pdf_ptr(img)->xref;
        epdf_doc = pdf_ptr(img)->doc;
        epdf_delete();
        break;
    case IMAGE_TYPE_PNG:
        xfclose(png_ptr(img)->io_ptr, cur_file_name);
        png_destroy_read_struct(&(png_ptr(img)), &(png_info(img)), NULL);
        break;
    case IMAGE_TYPE_JPG:
        xfclose(jpg_ptr(img)->file, cur_file_name);
        break;
#ifdef HAVE_TIFF
    case IMAGE_TYPE_TIF:
        TIFFClose(tif_ptr(img));
        break;
#endif
    default:
        pdftex_fail("unknown type of image");
    }
    xfree(img_name(img));
    img_type(img) = IMAGE_TYPE_NONE; /* mark as a deleted entry to be reused */
    return;
}

void img_free() 
{
    xfree(image_tab);
}
