#include "ptexlib.h"
#include "image.h"
#include <kpathsea/c-memstr.h>

image_entry *image_ptr, *image_tab = 0;
integer image_max;

int epdf_width;
int epdf_height;
int epdf_orig_x;
int epdf_orig_y;
int epdf_selected_page;
void *epdf_doc;
void *epdf_xref;

static integer new_image_entry(void)
{
    entry_room(image, 256);
    image_ptr->image_type = 0;
    return image_ptr++ - image_tab;
}

void img_free() 
{
    xfree(image_tab);
}

void deleteimage(integer img)
{
    switch (img_type(img)) {
    case IMAGE_TYPE_PNG:
        xfclose(png_ptr(img)->io_ptr, cur_file_name);
        png_destroy_read_struct(&(png_ptr(img)), &(png_info(img)), NULL);
        break;
    case IMAGE_TYPE_JPG:
        xfclose(jpg_info(img)->file, cur_file_name);
        break;
#ifdef HAVE_TIFF
    case IMAGE_TYPE_TIFF:
        TIFFClose((TIFF *) tiff_info(img)->file);
        break;
#endif
    case IMAGE_TYPE_PDF:
        epdf_xref = pdf_info(img)->xref;
        epdf_doc = pdf_info(img)->doc;
        epdf_delete();
        break;
    default:
        pdftex_fail("unknown type of image");
    }
    xfree(img_name(img));
    return;
}

integer imagewidth(integer img)
{
    switch (img_type(img)) {
    case IMAGE_TYPE_PNG:
        return png_info(img)->width;
    case IMAGE_TYPE_JPG:
        return jpg_info(img)->width;
#ifdef HAVE_TIFF
    case IMAGE_TYPE_TIFF:
        return tiff_info(img)->width;
#endif
    case IMAGE_TYPE_PDF:
        return pdf_info(img)->width;
    default:
        pdftex_fail("unknown type of image");
        return -1; /* never searched */
    }
}

integer imageheight(integer img)
{
    switch (img_type(img)) {
    case IMAGE_TYPE_PNG:
        return png_info(img)->height;
    case IMAGE_TYPE_JPG:
        return jpg_info(img)->height;
#ifdef HAVE_TIFF
    case IMAGE_TYPE_TIFF:
        return tiff_info(img)->height;
#endif
    case IMAGE_TYPE_PDF:
        return pdf_info(img)->height;
    default:
        pdftex_fail("unknown type of image");
        return -1; /* never searched */
    }
}

integer imagexres(integer img)
{
    switch (img_type(img)) {
    case IMAGE_TYPE_PNG:
        if (png_info(img)->valid & PNG_INFO_pHYs)
            return 0.0254*png_get_x_pixels_per_meter(png_ptr(img), 
                                                     png_info(img)) + 0.5;
        return 0;
    case IMAGE_TYPE_JPG:
        return jpg_info(img)->x_res;
#ifdef HAVE_TIFF
    case IMAGE_TYPE_TIFF:
        return tiff_info(img)->x_res;
#endif
    case IMAGE_TYPE_PDF:
        return 0;
    default:
        pdftex_fail("unknown type of image");
        return -1; /* never searched */
    }
}

integer imageyres(integer img)
{
    switch (img_type(img)) {
    case IMAGE_TYPE_PNG:
        if (png_info(img)->valid & PNG_INFO_pHYs)
            return 0.0254*png_get_y_pixels_per_meter(png_ptr(img),
                                                     png_info(img)) + 0.5;
        return 0;
    case IMAGE_TYPE_JPG:
        return jpg_info(img)->y_res;
#ifdef HAVE_TIFF
    case IMAGE_TYPE_TIFF:
        return tiff_info(img)->y_res;
#endif
    case IMAGE_TYPE_PDF:
        return 0;
    default:
        pdftex_fail("unknown type of image");
        return -1; /* never searched */
    }
}

boolean ispdfimage(integer img)
{
    return img_type(img) == IMAGE_TYPE_PDF;
}

boolean checkimageb(integer procset)
{
    return procset & IMAGE_COLOR_B;
}

boolean checkimagec(integer procset)
{
    return procset & IMAGE_COLOR_C;
}

boolean checkimagei(integer procset)
{
    return procset & IMAGE_COLOR_I;
}

void updateimageprocset(integer procset)
{
    pdfimageprocset |= procset;
}

integer epdforigx(integer img)
{
    return pdf_info(img)->orig_x;
}

integer epdforigy(integer img)
{
    return pdf_info(img)->orig_y;
}

integer readimage(integer page_num)
{
    char *image_suffix;
    integer img = new_image_entry();
    cur_file_name = makecstring(makenamestring());
    flushlaststring(last_tex_string);
    img_name(img) = kpse_find_file(cur_file_name, kpse_tex_format, true);
    if (img_name(img) == 0)
        pdftex_fail("cannot find image file");
    if ((image_suffix = rindex(cur_file_name, '.')) == 0)
        pdftex_fail("cannot find image file name extension");
    if (strcasecmp(image_suffix, ".pdf") == 0) {
        pdf_info(img) = xtalloc(1, pdf_image_struct);
        read_pdf_info(img_name(img), page_num);
        pdf_info(img)->width = epdf_width;
        pdf_info(img)->height = epdf_height;
        pdf_info(img)->orig_x = epdf_orig_x;
        pdf_info(img)->orig_y = epdf_orig_y;
        pdf_info(img)->selected_page = page_num;
        pdf_info(img)->xref = epdf_xref;
        pdf_info(img)->doc = epdf_doc;
        set_img_type(img, IMAGE_TYPE_PDF);
    }
    else if (strcasecmp(image_suffix, ".png") == 0) {
        read_png_info(img);
        set_img_type(img, IMAGE_TYPE_PNG);
    }
    else if (strcasecmp(image_suffix, ".jpg") == 0 ||
             strcasecmp(image_suffix, ".jpeg") == 0) {
        jpg_info(img) = xtalloc(1, JPG_IMAGE_INFO);
        switch (read_jpg_info(img)) {
        case 0:
            break;
        case 4:
            pdftex_fail("unsupported type of compression");
        default: 
            pdftex_fail("reading JPG image failed");
        }
        set_img_type(img, IMAGE_TYPE_JPG);
    }
#ifdef HAVE_TIFF
    else if (strcasecmp(image_suffix, ".tif") == 0 ||
             strcasecmp(image_suffix, ".tiff") == 0) {
        tiff_info(img) = XTALLOC(1, TIFF_IMAGE_INFO);
        switch (read_tiff_info(img)) {
        case 0:
            break;
        default:
            pdftex_fail("reading TIFF image failed");
        }
        set_img_type(img, IMAGE_TYPE_TIFF);
    }
#endif
    else 
        pdftex_fail("unknown type of image");
    cur_file_name = 0;
    return img;
}

integer imagetype(integer img)
{
    return (image_tab + img)->image_type;
}

void writeimage(integer img)
{
    tex_printf(" <%s", img_name(img));
    switch (img_type(img)) {
    case IMAGE_TYPE_PNG:
        write_png(img);
        break;
    case IMAGE_TYPE_JPG:
        write_jpg(img);
        break;
#ifdef HAVE_TIFF
    case IMAGE_TYPE_TIFF:
        write_tiff(img);
        break;
#endif
    case IMAGE_TYPE_PDF:
        epdf_xref = pdf_info(img)->xref;
        epdf_doc = pdf_info(img)->doc;
        epdf_selected_page = pdf_info(img)->selected_page;
        write_epdf();
        break;
    default:
        pdftex_fail("unknown type of image");
    }
    tex_printf(">");
}
