#include <stdlib.h>
#include <stddef.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <GString.h>
#include <gmem.h>
#include <gfile.h>
#include <config.h>
#include "Object.h"
#include "Stream.h"
#include "Array.h"
#include "Dict.h"
#include "XRef.h"
#include "Catalog.h"
#include "Page.h"
#include "GfxFont.h"
#include "PDFDoc.h"
#include "Params.h"
#include "Error.h"

#include "epdf.h"

#define obj_free(obj)  if (!obj.isNull())    obj.free()

struct IndirectObj {
    Ref ref;             // ref in original PDF
    integer newNum;      // new object number in output PDF
    IndirectObj *next;   // next entry in list of indirect objects
    GBool skip;          // to mark objects that are skipped (font file streams)
    integer fontnameNum; // object number of 1) BaseFont entry if the current
                         //     object is a Font dict; or 2) FontName entry if
                         //     the current object is a FontDescriptor dict; or
                         //     3) zero for other objects.
    integer encodingNum; // similar to fontnameNum (for a Font dict only)

};

struct UsedEncoding {
    integer encodingNum;
    GfxFont *font;
    UsedEncoding *next;
};

GBool printCommands = gFalse;
IndirectObj *indirectObjList;
UsedEncoding *encodingList;
static GBool isInit = gFalse;

static int addEncoding(GfxFont *gfont)
{
    UsedEncoding *n = new UsedEncoding[1];
    n->next = encodingList;
    encodingList = n;
    n->font = gfont;
    n->encodingNum = pdfnewobjnum();
    return n->encodingNum;
}

static int addIndirectObj(Ref r, integer fontfileNum, integer fontnameNum, 
                          GfxFont *gfont)
{
    IndirectObj *p, *q, *n;
    if (r.num == 0)
        pdftex_fail("pdf inclusion: invalid reference");
    n = new IndirectObj[1];
    n->ref = r; 
    n->next = 0;
    n->encodingNum = 0;
    n->fontnameNum = fontnameNum;
    if (indirectObjList == 0)
        indirectObjList = n;
    else {
        for (p = indirectObjList; p != 0; p = p->next) {
            if (p->ref.num == r.num && p->ref.gen == r.gen) {
                delete n;
                delete gfont;
                return p->newNum;
            }
            q = p;
        }
        q->next = n;
    }
    if (fontfileNum == 0) { // not a FontFile stream object, 
        n->newNum = pdfnewobjnum();
        n->skip = gFalse;
        if (gfont != 0)
            n->encodingNum = addEncoding(gfont);
    }
    else { // a FontFile stream object, not copy but get the object number only 
        n->newNum = fontfileNum;
        n->skip = gTrue;
    }
    return n->newNum;
}

static void copyObject(Object *);

static void copyDictEntry(Object *obj, int i)
{
    Object obj1;
    pdf_printf("/%s ", obj->dictGetKey(i));
    obj->dictGetValNF(i, &obj1);
    copyObject(&obj1);
    obj1.free();
    pdf_puts("\n");
}

static void copyDict(Object *obj)
{
    int i, l;
    if (!obj->isDict())
        pdftex_fail("pdf inclusion: invalid dict type <%s>", 
                    obj->getTypeName());
    for (i = 0, l = obj->dictGetLength(); i < l; ++i)
        copyDictEntry(obj, i);
}

static void copyFontDict(Object *obj, integer fontnameNum, integer encodingNum)
{
    int i, l;
    GBool isFontDict;
    char *key;
    if (!obj->isDict())
        pdftex_fail("pdf inclusion: invalid dict type <%s>", 
                    obj->getTypeName());
    isFontDict = obj->isDict("Font");
    for (i = 0, l = obj->dictGetLength(); i < l; ++i) {
        key = obj->dictGetKey(i);
        if ((isFontDict && strcmp("BaseFont", key) == 0) ||
            strcmp("FontName", key) == 0) {
            pdf_printf("/%s %i 0 R\n", key, (int)fontnameNum);
            continue;
        }
        if (isFontDict && strcmp("Encoding", key) == 0)
            continue;
        copyDictEntry(obj, i);
    }
    if (isFontDict && encodingNum != 0)
        pdf_printf("/Encoding %i 0 R\n", (int)encodingNum);
}

static void copyStream(Stream *str)
{
    int c;
    str->reset();
    while ((c = str->getChar()) != EOF)
        pdfout(c);
}

static void copyProcSet(Object *obj)
{
    int i, l;
    Object procset;
    if (!obj->isArray())
        pdftex_fail("pdf inclusion: invalid ProcSet array type <%s>", 
                    obj->getTypeName());
    pdf_puts("/ProcSet [ ");
    for (i = 0, l = obj->arrayGetLength(); i < l; ++i) {
        obj->arrayGet(i, &procset);
        if (!procset.isName())
            pdftex_fail("pdf inclusion: invalid ProcSet entry type <%s>", 
                        procset.getTypeName());
        pdf_printf("/%s ", procset.getName());
        procset.free();
    }
    pdf_puts("]\n");
}

static void copyFont(char *tag, Object *fontRef)
{
    Object fontdict, subtype, basefont, encodingRef, fontdescRef, fontdesc,
           fontfileRef, fontfile, charset;
    GfxFont *gfont;
    int texfont;
    fontdict.initNull();
    subtype.initNull();
    basefont.initNull();
    fontdescRef.initNull();
    fontdesc.initNull();
    fontfileRef.initNull();
    fontfile.initNull();
    encodingRef.initNull();
    fontRef->fetch(&fontdict);
    if (!fontdict.isDict())
        pdftex_fail("pdf inclusion: invalid font dict type <%s>", 
                    fontdict.getTypeName());
    fontdict.dictLookup("Subtype", &subtype);
    if (!subtype.isName())
        pdftex_fail("pdf inclusion: invalid font Subtype entry type <%s>", 
                    subtype.getTypeName());
    /* only handle Type1 and TrueType fonts; others will be copied */
    if (strcmp(subtype.getName(), "Type1") != 0 &&
        strcmp(subtype.getName(), "TrueType") != 0) {
            pdf_printf("/%s ", tag);
            copyObject(fontRef);
            goto free_objects;
    }
    fontdict.dictLookup("BaseFont", &basefont);
    if (!basefont.isName())
        pdftex_fail("pdf inclusion: invalid font BaseFont entry type <%s>", 
                    basefont.getTypeName());
    texfont = lookup_fbname(basefont.getName());
    if (texfont >= 0 && 
        fontdict.dictLookupNF("FontDescriptor", &fontdescRef) && 
        fontdescRef.isRef() && fontdescRef.fetch(&fontdesc) &&
        fontdesc.dictLookupNF("FontFile", &fontfileRef) &&
        fontfileRef.isRef() && fontfileRef.fetch(&fontfile)) {
        if (fontdesc.dictLookup("CharSet", &charset) && 
            charset.isString() && is_subsetable(texfont)) {
            mark_glyphs(texfont, charset.getString()->getCString());
            charset.free();
        }
        else
            embed_whole_font(texfont);
        // mark the font file stream not to be copied
        addIndirectObj(fontfileRef.getRef(), get_fontfile_num(texfont), 0, 0);
        // replace the original font name in font descriptor by the name
        //     generated by pdftex
        addIndirectObj(fontdescRef.getRef(), 0, get_fontname_num(texfont), 0);
    }
    // similarly replace the original font name and encoding in Font dict by
    //      the data generated by pdftex
    if (!fontfile.isNull()) {
        gfont = new GfxFont(tag, fontRef->getRef(), fontdict.getDict());
        pdf_printf("/%s %d 0 R ", tag, 
                   addIndirectObj(fontRef->getRef(), 0, 
                                  get_fontname_num(texfont), 
                                  gfont));
    }
    else
        pdf_printf("/%s %d 0 R ", tag, 
                   addIndirectObj(fontRef->getRef(), 0, 0, 0));
free_objects:
    obj_free(fontdict);
    obj_free(subtype);
    obj_free(basefont);
    obj_free(fontdescRef);
    obj_free(fontdesc);
    obj_free(fontfileRef);
    obj_free(fontfile);
    obj_free(encodingRef);
}

static void copyFontResources(Object *obj)
{
    Object fontRef;
    int i, l;
    if (!obj->isDict())
        pdftex_fail("pdf inclusion: invalid font resources dict type <%s>", 
                    obj->getTypeName());
    pdf_puts("/Font << ");
    for (i = 0, l = obj->dictGetLength(); i < l; ++i) {
        obj->dictGetValNF(i, &fontRef);
        if (fontRef.isRef())
            copyFont(obj->dictGetKey(i), &fontRef);
        else
            pdftex_fail("pdf inclusion: invalid font indirect reference type <%s>", 
                        fontRef.getTypeName());
        fontRef.free();
    }
    pdf_puts(">>\n");
}

static void copyOtherResources(Object *obj, char *key)
{
    Object obj1;
    int i, l;
    if (!obj->isDict())
        pdftex_fail("pdf inclusion: invalid other resources dict type <%s>", 
                    obj->getTypeName());
    pdf_printf("/%s << ", key);
    for (i = 0, l = obj->dictGetLength(); i < l; ++i) {
        obj->dictGetValNF(i, &obj1);
        if (obj1.isRef())
            pdf_printf("/%s %d 0 R ", obj->dictGetKey(i), 
                       addIndirectObj(obj1.getRef(), 0, 0, 0));
        else 
            copyObject(&obj1);
        obj1.free();
    }
    pdf_puts(">>\n");
}

static void copyObject(Object *obj)
{
    Object obj1;
    int  i, l, c;
    Ref r;
    char *p;
    if (obj->isBool()) {
        pdf_printf("%s", obj->getBool() ? "true" : "false");
    }
    else if (obj->isInt()) {
        pdf_printf("%i", obj->getInt());
    }
    else if (obj->isReal()) {
        pdf_printf("%g", obj->getReal());
    }
    else if (obj->isNum()) {
        pdf_printf("%g", obj->getNum());
    }
    else if (obj->isString()) {
        pdf_puts("(");
        for (p = obj->getString()->getCString(); *p != 0; p++) {
            c = *p;
            if (c == '(' || c == ')' || c == '\\')
                pdf_printf("\\%c", c);
            else if (c < 0x20 || c > 0x7F)
                pdf_printf("\\%03o", c);
            else
                pdfout(c);
        }
        pdf_puts(")");
    }
    else if (obj->isName()) {
        pdf_puts("/");
        for (p = obj->getName(); *p; p++) {
            switch (*p) {
            case '(':
            case ')':
            case '<':
            case '>':
            case '[':
            case ']':
            case '{':
            case '}':
            case '/':
            case '#':
            case ' ':
                pdf_printf("#%.2X", *p);
                break;
            default:
                pdfout(*p);
            }
        }
    }
    else if (obj->isNull()) {
        pdf_puts("null");
    }
    else if (obj->isArray()) {
        pdf_puts("[");
        for (i = 0, l = obj->arrayGetLength(); i < l; ++i) {
            obj->arrayGetNF(i, &obj1);
            if (!obj1.isName())
                pdf_puts(" ");
            copyObject(&obj1);
            obj1.free();
        }
        pdf_puts("]");
    }
    else if (obj->isDict()) {
        pdf_puts("<<\n");
        copyDict(obj);
        pdf_puts(">>");
    }
    else if (obj->isStream()) {
        obj1.initDict(obj->getStream()->getDict());
        obj->getStream()->getDict()->incRef();
        pdf_puts("<<\n");
        copyDict(&obj1);
        obj1.free();
        pdf_puts(">>\n");
        pdf_puts("stream\n");
        copyStream(obj->getStream()->getBaseStream());
        pdf_puts("endstream");
    }
    else if (obj->isRef()) {
        r = obj->getRef();
        if (r.num == 0) {
            pdftex_warn("pdf inclusion: reference to invalid object was replaced by <null>");
            pdf_puts("null");
        }
        else
            pdf_printf("%d 0 R", addIndirectObj(r, 0, 0, 0));
    }
    else {
        pdftex_fail("pdf inclusion: type <%s> cannot be copied", 
                    obj->getTypeName());
    }
}

static void writeRefs()
{
    Object obj1;
    IndirectObj *r, *n;
    for (r = indirectObjList; r != 0; r = r->next) {
        if (r->skip)
            continue;
        zpdfbeginobj(r->newNum);
        xref->fetch(r->ref.num, r->ref.gen, &obj1);
        if (r->fontnameNum != 0) {
            pdf_puts("<<\n");
            copyFontDict(&obj1, r->fontnameNum, r->encodingNum);
            pdf_puts(">>");
        }
        else
            copyObject(&obj1);
        pdf_puts("\nendobj\n");
        obj1.free();
    }
    for (r = indirectObjList; r != 0; r = n) {
        n = r->next;
        delete r;
    }
}

static void writeEncodings()
{
    UsedEncoding *n, *r;
    char *glyphNames[MAX_CHAR_CODE + 1], *s;
    int i;
    for (r = encodingList; r != 0; r = r->next) {
        for (i = 0; i <= MAX_CHAR_CODE; i++)
            if ((s = r->font->getCharName(i)) != 0)
                glyphNames[i] = s;
            else
                glyphNames[i] = notdef;
        write_enc(glyphNames, r->encodingNum);
    }
    for (r = encodingList; r != 0; r = n) {
        n = r->next;
        delete r->font;
        delete r;
    }
}

integer read_pdf_info(char *image_name, integer page_num)
{
    PDFDoc *doc;
    GString *docName;
    Page *page;
    Object contents, obj, resources;
    int n;
    // initialize
    if (!isInit) {
        initParams(xpdfConfigFile);
        errorInit();
        isInit = gTrue;
    }
    // open PDF file
    xref = 0;
    docName = new GString(image_name);
    doc = new PDFDoc(docName);
    if (!doc->isOk() || !doc->okToPrint())
        return -1;
    epdf_doc = (void *)doc;
    epdf_xref = (void *)xref;
    n = doc->getCatalog()->getNumPages();
    if (page_num <= 0 || page_num > n)
        pdftex_fail("pdf inclusion: required page does not exists <%i>", n);
    // get the required page
    page = doc->getCatalog()->getPage(page_num);
    if (page->isCropped()) {
        epdf_width = (int)(page->getCropX2() - page->getCropX1());
        epdf_height = (int)(page->getCropY2() - page->getCropY1());
        epdf_orig_x = (int)page->getCropX1();
        epdf_orig_y = (int)page->getCropY1();
    }
    else {
        epdf_width = (int)(page->getX2() - page->getX1());
        epdf_height = (int)(page->getY2() - page->getY1());
        epdf_orig_x = (int)page->getX1();
        epdf_orig_y = (int)page->getY1();
    }
    return 0;
}

void write_epdf(integer img)
{
    Page *page;
    Object contents, obj1, obj2;
    char *key;
    int i, l;
    xref = (XRef *)epdf_xref;
    page = ((PDFDoc *)epdf_doc)->getCatalog()->getPage(epdf_selected_page);
    indirectObjList = 0;
    encodingList = 0;
    // write the Page header
    pdf_puts("/Type /XObject\n");
    pdf_puts("/Subtype /Form\n");
    pdf_puts("/FormType 1\n");
    pdf_puts("/Matrix [1 0 0 1 0 0]\n");
    if (page->isCropped())
        pdf_printf("/BBox [%i %i %i %i]\n",
                  (int)page->getCropX1(),
                  (int)page->getCropY1(),
                  (int)page->getCropX2(),
                  (int)page->getCropY2());
    else 
        pdf_printf("/BBox [%i %i %i %i]\n",
                  (int)page->getX1(),
                  (int)page->getY1(),
                  (int)page->getX2(),
                  (int)page->getY2());
    // write the Resources dictionary
    obj1.initDict(page->getResourceDict());
    page->getResourceDict()->incRef();
    if (!obj1.isDict())
        pdftex_fail("pdf inclusion: invalid resources dict type <%s>", 
                    obj1.getTypeName());
    pdf_puts("/Resources <<\n");
    for (i = 0, l = obj1.dictGetLength(); i < l; ++i) {
        obj1.dictGetVal(i, &obj2);
        key = obj1.dictGetKey(i);
        if (strcmp("Font", key) == 0)
            copyFontResources(&obj2);
        else if (strcmp("ProcSet", key) == 0)
            copyProcSet(&obj2);
        else
            copyOtherResources(&obj2, key);
        obj2.free();
    }
    obj1.free();
    pdf_puts(">>\n");
    // write the page contents
    page->getContents(&contents);
    if (contents.isStream()) {
        obj1.initDict(contents.getStream()->getDict());
        contents.getStream()->getDict()->incRef();
        copyDict(&obj1);
        pdf_puts(">>\nstream\n");
        copyStream(contents.getStream()->getBaseStream());
        pdf_puts("endstream\nendobj\n");
        obj1.free();
    }
    else if (contents.isArray()) {
        pdfbeginstream();
        for (i = 0, l = contents.arrayGetLength(); i < l; ++i) {
            contents.arrayGet(i, &obj1);
            copyStream(obj1.getStream());
            obj1.free();
        }
        pdfendstream();
    }
    else
        pdftex_fail("pdf inclusion: invalid page contents type <%s>", 
                    contents.getTypeName());
    contents.free();
    // write out all indirect objects
    writeRefs();
    // write out all used encodings
    writeEncodings();
}

void epdf_delete()
{
    xref = (XRef *)epdf_xref;
    delete (PDFDoc *)epdf_doc;
}

void epdf_check_mem()
{
    if (isInit) {
        freeParams();
        Object::memCheck(errFile);
        gMemReport(errFile);
    }
}
