#include "ptexmac.h"
typedef unsigned char byte;


#ifdef pdfTeX /* use with pdfTeX */

#undef pdfTeX /* to avoid warning about redefining pdfTeX in pdftexd.h */
#include "ptexlib.h"           
                    

#define pdftex_fail             pdftex_fail
#define pdftex_warn             pdftex_warn
#define t1_open()           \
    open_input(&t1_file, kpse_type1_format, FOPEN_RBIN_MODE)
#define t1_getchar()        xgetc(t1_file)
#define t1_eof()            feof(t1_file)
#define t1_putchar(c)       pdfout(c)
#define t1_close()          xfclose(t1_file, cur_file_name)
#define t1_log(s)           tex_printf(s)

#define save_offset()       pdfsaveoffset = pdfoffset()
#define get_length1()       t1_length1 = pdfoffset() - pdfsaveoffset
#define get_length2()       t1_length2 = pdfoffset() - pdfsaveoffset
#define get_length3()       t1_length3 = pdfoffset() - pdfsaveoffset
#define external_enc()      enc_tab[fm_cur->encoding].glyph_names
#define update_cur_enc()    update_enc(tex_font, t1_glyph_names) 
#define is_used_char(c)     pdfcharmarked(tex_font, c)

integer t1_length1, t1_length2, t1_length3;

#else /* use with dvips */

#include "dvips.h"
#include <kpathsea/c-vararg.h>
#include <kpathsea/c-proto.h>

#define t1_getchar()        getc(t1_file)
#define t1_putchar(c)       fputc(c, bitfile)
#define t1_eof()            feof(t1_file)
#define t1_close()          xfclose(t1_file, 0)
#define t1_log(s)           fprintf(stderr, s)
#define t1_scan_keys()

#define save_offset()
#define get_length1()
#define get_length2()
#define get_length3()
#define external_enc()      ext_glyph_names
#define update_cur_enc()
#define t1_mark_extra_glyphs()

#undef  fm_extend
#undef  fm_slant
#undef  is_reencoded
#undef  is_subsetted
#undef  is_included

#define fm_extend(fm)       0
#define fm_slant(fm)        0
#define is_reencoded(fm)    (enc_file != 0)
#define is_included(fm)     true
#define is_subsetted(fm)    true
#define is_used_char(c)     (grid[c] == 1)

extern FILE *bitfile ;
static char *cur_file_name;
static char *enc_file;
static unsigned char *grid;
static char *ext_glyph_names[MAX_CHAR_CODE + 1];                                        
static char print_buf[PRINTF_BUF_SIZE];

void pdftex_fail(char *fmt,...)
{
    va_list args;
    va_start(args, fmt);
    /*
    fprintf(stderr, "\nError: %s", program_invocation_name);
    if (cur_file_name)
        tex_printf(" (file %s)", cur_file_name);
    tex_printf(": ");
    */
    vsprintf(print_buf, fmt, args);
    va_end(args);
    exit(-1);
}

void pdftex_warn(char *fmt,...)
{
    va_list args;
    va_start(args, fmt);
    /*
    println();
    tex_printf("Warning: %s", program_invocation_name);
    if (cur_file_name)
        tex_printf(" (file %s)", cur_file_name);
    tex_printf(": ");
    print(maketexstring(print_buf));
    flushstring();
    */
    vsprintf(print_buf, fmt, args);
    va_end(args);
}

#endif /* pdfTeX */


char notdef[] = ".notdef";

static char *standard_glyph_names[] = {
notdef, notdef, notdef, notdef, notdef, notdef, notdef, notdef, notdef,
notdef, notdef, notdef, notdef, notdef, notdef, notdef, notdef, notdef,
notdef, notdef, notdef, notdef, notdef, notdef, notdef, notdef, notdef,
notdef, notdef, notdef, notdef, notdef, "space", "exclam", "quotedbl",
"numbersign", "dollar", "percent", "ampersand", "quoteright", "parenleft",
"parenright", "asterisk", "plus", "comma", "hyphen", "period", "slash",
"zero", "one", "two", "three", "four", "five", "six", "seven", "eight",
"nine", "colon", "semicolon", "less", "equal", "greater", "question", "at",
"A", "B", "C", "D", "E", "F", "G", "H", "I", "J", "K", "L", "M", "N", "O",
"P", "Q", "R", "S", "T", "U", "V", "W", "X", "Y", "Z", "bracketleft",
"backslash", "bracketright", "asciicircum", "underscore", "quoteleft", "a",
"b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p",
"q", "r", "s", "t", "u", "v", "w", "x", "y", "z", "braceleft", "bar",
"braceright", "asciitilde", notdef, notdef, notdef, notdef, notdef, notdef,
notdef, notdef, notdef, notdef, notdef, notdef, notdef, notdef, notdef,
notdef, notdef, notdef, notdef, notdef, notdef, notdef, notdef, notdef,
notdef, notdef, notdef, notdef, notdef, notdef, notdef, notdef, notdef,
notdef, "exclamdown", "cent", "sterling", "fraction", "yen", "florin",
"section", "currency", "quotesingle", "quotedblleft", "guillemotleft",
"guilsinglleft", "guilsinglright", "fi", "fl", notdef, "endash", "dagger",
"daggerdbl", "periodcentered", notdef, "paragraph", "bullet",
"quotesinglbase", "quotedblbase", "quotedblright", "guillemotright",
"ellipsis", "perthousand", notdef, "questiondown", notdef, "grave", "acute",
"circumflex", "tilde", "macron", "breve", "dotaccent", "dieresis", notdef,
"ring", "cedilla", notdef, "hungarumlaut", "ogonek", "caron", "emdash",
notdef, notdef, notdef, notdef, notdef, notdef, notdef, notdef, notdef,
notdef, notdef, notdef, notdef, notdef, notdef, notdef, "AE", notdef,
"ordfeminine", notdef, notdef, notdef, notdef, "Lslash", "Oslash", "OE",
"ordmasculine", notdef, notdef, notdef, notdef, notdef, "ae", notdef, notdef,
notdef, "dotlessi", notdef, notdef, "lslash", "oslash", "oe", "germandbls",
notdef, notdef, notdef, notdef
};

char **t1_glyph_names;
char *t1_builtin_glyph_names[MAX_CHAR_CODE + 1];                                        
static boolean read_encoding_only;
static int t1_encoding;

typedef char * extra_glyphs_entry;
static char **extra_glyphs_ptr, **extra_glyphs_tab;
static int extra_glyphs_max;

#define T1_BUF_SIZE   4096

#define ENC_STANDARD  0
#define ENC_BUILTIN   1

#define CS_HSTEM            1
#define CS_VSTEM            3
#define CS_VMOVETO          4
#define CS_RLINETO          5
#define CS_HLINETO          6
#define CS_VLINETO          7
#define CS_RRCURVETO        8
#define CS_CLOSEPATH        9
#define CS_CALLSUBR         10
#define CS_RETURN           11
#define CS_ESCAPE           12
#define CS_HSBW             13
#define CS_ENDCHAR          14
#define CS_RMOVETO          21
#define CS_HMOVETO          22
#define CS_VHCURVETO        30
#define CS_HVCURVETO        31
#define CS_1BYTE_MAX        (CS_HVCURVETO + 1)

#define CS_DOTSECTION       CS_1BYTE_MAX + 0
#define CS_VSTEM3           CS_1BYTE_MAX + 1
#define CS_HSTEM3           CS_1BYTE_MAX + 2
#define CS_SEAC             CS_1BYTE_MAX + 6
#define CS_SBW              CS_1BYTE_MAX + 7
#define CS_DIV              CS_1BYTE_MAX + 12
#define CS_CALLOTHERSUBR    CS_1BYTE_MAX + 16
#define CS_POP              CS_1BYTE_MAX + 17
#define CS_SETCURRENTPOINT  CS_1BYTE_MAX + 33
#define CS_2BYTE_MAX        (CS_SETCURRENTPOINT + 1)
#define CS_MAX              CS_2BYTE_MAX

typedef struct {
    byte nargs;     /* number of arguments */
    boolean bottom; /* take arguments from bottom of stack? */
    boolean clear;  /* clear stack? */
    boolean valid;
} cc_entry; /* CharString Command */

typedef struct {
    char *name;             /* glyph name (or notdef for Subrs entry) */
    byte *data;
    unsigned short len;     /* length of the whole string */
    unsigned short cslen;   /* length of the encoded part of the string */
    boolean used;
    boolean invalid;
} cs_entry;

#define set_subr_invalid(P) (P)->name = 0
#define set_subr_valid(P)   (P)->name = notdef
#define is_subr_valid(P)    (P)->name != 0

static unsigned short t1_dr, t1_er;
static unsigned short t1_c1 = 52845, t1_c2 = 22719;
static unsigned short t1_cslen, t1_lenIV;
static char t1_line[T1_BUF_SIZE], t1_buf[T1_BUF_SIZE], *t1_line_ptr;

static char *cs_start;

static cs_entry *cs_tab, *cs_ptr, *cs_notdef;
static char *cs_dict_start, *cs_dict_end;
static int cs_count, cs_size, cs_size_pos;

static cs_entry *subr_tab;
static char *subr_array_start, *subr_array_end;
static int subr_max, subr_size, subr_size_pos;

static boolean t1_in_eexec, t1_pfa, t1_cs, t1_scan;
static long t1_block_length;
static FILE *t1_file;

#define str_prefix(s1, s2)  (strncmp(s1, s2, strlen(s2)) == 0)
#define t1_prefix(s)        str_prefix(t1_line, s)
#define t1_charstrings()    strstr(t1_line, "/CharStrings")
#define t1_subrs()          t1_prefix("/Subrs")

#define t1_printf(s) do {                                  \
    sprintf(t1_line, s);                                   \
    t1_line_ptr = strend(t1_line);                         \
    append_eol(t1_line_ptr, t1_line, T1_BUF_SIZE);         \
    t1_putline();                                          \
} while (0)
    
#define AND ,

static void t1_check_pfa(void)
{
    int c = t1_getchar();
    if (c != 128)
        t1_pfa = true;
    else 
        t1_pfa = false;
    ungetc(c, t1_file);
}

static int t1_getbyte(void)
{
    int c = t1_getchar();
    if (t1_pfa)
        return c;
    if (t1_block_length == 0) {
        if (c != 128)
            pdftex_fail("invalid marker");
        c = t1_getchar();
        if (c == 3)
            return EOF;
        t1_block_length = t1_getchar() & 0xff;
        t1_block_length |= (t1_getchar() & 0xff) << 8;
        t1_block_length |= (t1_getchar() & 0xff) << 16;
        t1_block_length |= (t1_getchar() & 0xff) << 24;
        c = t1_getchar();
    }
    t1_block_length--;
    return c;
}

static int hexval(int c)
{
    if (c >= 'A' && c <= 'F')
        return c - 'A' + 10;
    else if (c >= 'a' && c <= 'f')
        return c - 'a' + 10;
    else if (c >= '0' && c <= '9')
        return c - '0';
    else
        return 0;
}

static byte edecrypt(byte cipher)
{
    byte plain;
    if (t1_pfa) {
        while (cipher == 10 || cipher == 13)
            cipher = t1_getbyte();
        cipher = (hexval(cipher) << 4) + hexval(t1_getbyte());
    }
    plain = (cipher^(t1_dr >> 8));
    t1_dr = (cipher + t1_dr)*t1_c1 + t1_c2;
    return plain;
}

static byte cdecrypt(byte cipher, unsigned short *cr)
{
    byte plain = (cipher^(*cr >> 8));
    *cr = (cipher + *cr)*t1_c1 + t1_c2;
    return plain;
}

static byte eencrypt(byte plain)
{
    byte cipher = (plain^(t1_er >> 8));
    t1_er = (cipher + t1_er)*t1_c1 + t1_c2;
    return cipher;
}

static char *eol(char *s)
{
    char *p = strend(s);
    if (p - s > 1 && p[-1] != 10) {
        *p++ = 10;
        *p = 0;
    }
    return p;
}

static boolean t1_suffix(char *s) 
{
    char *s1 = t1_line_ptr - 1, 
         *s2 = strend(s) - 1;
    if (*s1 == 10)
        s1--;
    while (s1 >= t1_line && s2 >= s) {
        if (*s1-- != *s2--)
            return false;
    }
    return s1 >= t1_line - 1;
}

static void t1_getline(void) 
{
    int c, l;
    char *p;
restart:
    if (t1_eof())
        pdftex_fail("unexpected end of file");
    t1_line_ptr = t1_line;
    t1_cslen = 0;
    c = t1_getbyte();
    while (!t1_eof()) {
        if (t1_in_eexec) 
            c = edecrypt(c);
        append_char_to_buf(c, t1_line_ptr, t1_line, T1_BUF_SIZE);
        if (c == 10)
            break;
        if (t1_cs && t1_cslen == 0 && 
            (t1_line_ptr - t1_line > 4) && 
            (t1_suffix(" RD ") || t1_suffix(" -| ")))
        {
            p = t1_line_ptr - 5;
            while (*p !=  ' ')
                p--;
            t1_cslen = l = atoi(p + 1);
            cs_start = t1_line_ptr;
            check_buf(t1_line_ptr - t1_line + l, T1_BUF_SIZE);
            while (l-- > 0)
                *t1_line_ptr++ = edecrypt(t1_getbyte());
        }
        c = t1_getbyte();
    }
    append_eol(t1_line_ptr, t1_line, T1_BUF_SIZE);
    if (t1_line_ptr - t1_line <= 1)
        goto restart;
}

static void t1_putline(void) 
{
    char *p = t1_line;
    if (t1_line_ptr - t1_line <= 1)
        return;
    if (t1_in_eexec) 
        while (p < t1_line_ptr)
            t1_putchar(eencrypt(*p++));
    else 
        while (p < t1_line_ptr)
            t1_putchar(*p++);
}

static void t1_init_params(char *open_name_prefix) 
{
    t1_log(open_name_prefix);
    t1_log(cur_file_name);
    t1_lenIV = 4;
    t1_dr = 55665;
    t1_er = 55665;
    t1_in_eexec = false;
    t1_cs = false;
    t1_scan = true;
    t1_block_length = 0;
    t1_check_pfa();
}


#ifdef pdfTeX
static void t1_modify_fm(void)
{
 /*
  * font matrix is given as six numbers a0..a5, which stands for the matrix
  * 
  *           a0 a1 0
  *     M =   a2 a3 0
  *           a4 a5 1
  * 
  * ExtendFont is given as
  * 
  *           e 0 0
  *     E =   0 1 0
  *           0 0 1
  * 
  * SlantFont is given as
  * 
  *           1 0 0
  *     S =   s 1 0
  *           0 0 1
  * 
  * and the final transformation is
  * 
  *                    e*a0        e*a1       0
  *     F =  E.S.M  =  s*e*a0+a2   s*e*a1+a3  0
  *                    a4          a5         1
  */
    float e, s, a[6], b[6];
    int i, c;
    char *p, *q;
    if ((p = strchr(t1_line, '[')) == 0)
        if ((p = strchr(t1_line, '{')) == 0) {
            remove_eol(p, t1_line);
            pdftex_fail("FontMatrix: an array expected: `%s'", t1_line);
        }
    c = *p; /* save the character '[' resp. '{' */
    strncpy(t1_buf, t1_line, p - t1_line + 1);
    if (sscanf(p + 1, "%f %f %f %f %f %f", 
                    a, a + 1, a + 2, a + 3, a + 4, a + 5) != 6) {
        remove_eol(p, t1_line);
        pdftex_fail("FontMatrix: six numbers expected: `%s'", t1_line);
    }
    if (fm_extend(fm_cur) != 0)
        e = fm_extend(fm_cur)*1E-3;
    else
        e = 1;
    s = fm_slant(fm_cur)*1E-3;
    b[0] = e*a[0];
    b[1] = e*a[1];
    b[2] = s*e*a[0] + a[2];
    b[3] = s*e*a[1] + a[3];
    b[4] = a[4];
    b[5] = a[5];
    q = t1_buf + (p - t1_line + 1);
    for (i = 0; i < 6; i++) {
        sprintf(q, "%G ", b[i]);
        q = strend(q);
    }
    if (c == '[') {
        while (*p != ']' && *p != 0)
            p++;
    }
    else {
        while (*p != '}' && *p != 0)
            p++;
    }
    if (*p == 0) {
        remove_eol(p, t1_line);
        pdftex_fail("FontMatrix: cannot find the corresponding character to '%c': `%s'",  c, t1_line);
    }
    strcpy(q, p);
    strcpy(t1_line, t1_buf);
    t1_line_ptr = eol(t1_line);
}

static void t1_modify_italic(void)
{
    float a;
    char buf[1024], *p, *r;
    if (fm_slant(fm_cur) == 0)
        return;
    p = strchr(t1_line, ' ');
    strncpy(t1_buf, t1_line, p - t1_line + 1);
    sscanf(p + 1, "%f", &a);
    for (r = p + 1; *r != ' ' && *r != 10; r++);
    a -= atan(fm_slant(fm_cur)*1E-3)*(180/M_PI);
    sprintf(t1_buf + (p - t1_line + 1), "%.2g", a);
    strcpy(strend(t1_buf), r);
    strcpy(t1_line, t1_buf);
    t1_line_ptr = eol(t1_line);
    font_keys[ITALIC_ANGLE_CODE].value.i = (a > 0) ? a + 0.5 : a - 0.5;
    font_keys[ITALIC_ANGLE_CODE].valid = true;
}

static boolean t1_open_fontfile(char *open_name_prefix)
{
    if (fm_cur->ex_name != 0) {
        get_texname(fm_cur->ex_name);
        if (t1_open()) /* found mm instance */
            goto open_ok;
    }
    if (fm_cur->found)
        t1_file = xfopen(cur_file_name = fm_cur->ff_name, FOPEN_RBIN_MODE);
    else {
        get_texname(fm_cur->ff_name);
        if (!t1_open()) {
            pdftex_warn("cannot open Type 1 font file for reading");
            return false;
        }
        fix_ffname(fm_cur, cur_file_name = nameoffile + 1);
    }
    if (pdfexpandfont[f] != 0) { /* use ExtendFont to simulate MM instance */
        if (fm_extend(fm_cur) == 0)
            fm_extend(fm_cur) = 1000;
        fm_extend(fm_cur) = xnoverd(fm_extend(fm_cur), 1000 + pdfexpandfont[f], 1000);
    }
open_ok:
    t1_init_params(open_name_prefix);
    fontfile_found = true;
    return true;
}

void static t1_scan_keys()
{
    int i, k;
    char *p, *q, *r;
    key_entry *key;
    if (fm_extend(fm_cur) != 0 || fm_slant(fm_cur) != 0) {
        if (strncmp(t1_line + 1, "UniqueID", strlen("UniqueID")) == 0) {
            t1_line_ptr = t1_line; /* ignore UniqueID */
            return;
        }
        if (strncmp(t1_line + 1, "FontMatrix", strlen("FontMatrix")) == 0) {
            t1_modify_fm();
            return;
        }
        if (strncmp(t1_line + 1, "ItalicAngle", strlen("ItalicAngle")) == 0) {
            t1_modify_italic();
            return;
        }
    }
    for (key = font_keys; key - font_keys  < MAX_KEY_CODE; key++)
        if (strncmp(t1_line + 1, key->t1name, strlen(key->t1name)) == 0)
          break;
    if (key - font_keys == MAX_KEY_CODE)
        return;
    key->valid = true;
    p = t1_line + strlen(key->t1name) + 1;
    skip(p, ' ');
    if ((k = key - font_keys) == FONTNAME_CODE) {
        if (*p != '/') {
            remove_eol(p, t1_line);
            pdftex_fail("a name expected: `%s'", t1_line);
        }
        r = ++p; /* skip the slash */
        for (q = t1_buf; *p != ' ' && *p != 10; *q++ = *p++);
        *q = 0;
        if (fm_extend(fm_cur) != 0) {
            sprintf(q, "-Extend_%i", (int)fm_extend(fm_cur));
        }
        if (fm_slant(fm_cur) != 0) {
            sprintf(q, "-Slant_%i", (int)fm_slant(fm_cur));
        }
        key->value.s = xstrdup(t1_buf);
        if (is_included(fm_cur) && is_subsetted(fm_cur)) {
            strcpy(t1_buf, p);
            sprintf(r, "%s+%s%s", fm_cur->subset_tag, key->value.s, t1_buf);
            t1_line_ptr = eol(r);
        }
        return;
    }
    if ((k == STEMV_CODE ||  k == FONTBBOX1_CODE) &&
        (*p == '[' || *p == '{'))
        p++;
    if (k == FONTBBOX1_CODE) {
        for (i = 0; i < 4; i++) {
            key[i].value.i = atoi(p);
            skip(p, ' ');
            skip(p, '-');
            while (isdigit(*p))
                p++;
        }
        return;
    }
    key->value.i = atoi(p);
}

boolean t1_read_enc(fm_entry *fm)
{
    read_encoding_only = true;
    fm_cur = fm;
    if (!t1_open_fontfile("{"))
        return false;
    fix_ffname(fm_cur, cur_file_name = nameoffile + 1);
    while (!t1_prefix("/Encoding"))
        t1_getline();
    t1_check_predef_enc();
    if (t1_encoding == ENC_BUILTIN)
        t1_builtin_enc();
    t1_close_font_file("}");
    return true;
}

static void t1_mark_extra_glyphs(void)
{
    int i;
    char *charset = fm_tab[pdffontmap[tex_font]].charset;
    char *g, *s, *r;
    extra_glyphs_entry *p;
    extra_glyphs_ptr = extra_glyphs_tab = 0;
    extra_glyphs_max = 0;
    if (charset == 0)
        return;
    g = s = charset + 1; /* skip the first '/' */
    r = strend(g);
    while (g < r) {
        while (*s != '/' && s < r)
            s++;
        *s = 0; /* terminate g by rewriting '/' to 0 */
        for (i = 0; i <= MAX_CHAR_CODE; i++)
            if (t1_glyph_names[i] != notdef && 
                strcmp(t1_glyph_names[i], g) == 0)  {
                pdfmarkchar(tex_font, i);
                break;
            }
        if (i > MAX_CHAR_CODE) {
            for (p = extra_glyphs_tab; p < extra_glyphs_ptr; p++)
                if (strcmp(*p, g) == 0)
                    break;
            if (p == extra_glyphs_ptr) {
                entry_room(extra_glyphs, 256);
                *extra_glyphs_ptr++ = *p;
            }
        }
        g = s + 1;
    }
}

static void t1_scan_only(void)
{
    int i;
    do {
        t1_getline();
        t1_scan_param();
    } while (!t1_prefix("currentfile eexec"));
    t1_in_eexec = true;
    for (i = 0; i < 4; i++)
        edecrypt(t1_getbyte());
    do {
        t1_getline();
        t1_scan_param();
    } while (!(t1_charstrings() || t1_subrs()));
}

static void t1_include(void)
{
    pdfsaveoffset = pdfoffset();
    do {
        t1_getline();
        t1_scan_param();
        t1_putline();
    } while (!t1_prefix("currentfile eexec"));
    t1_length1 = pdfoffset() - pdfsaveoffset;
    t1_eexec_init();
    do {
        t1_getline();
        t1_scan_param();
        t1_putline();
    } while (!(t1_charstrings() || t1_subrs()));
    t1_cs = true;
    do {
        t1_getline();
        t1_putline();
    } while (!t1_suffix("mark currentfile closefile"));
    t1_length2 = pdfoffset() - pdfsaveoffset;
    pdfsaveoffset = pdfoffset();
    t1_in_eexec = false;
    t1_cs = false;
    do {
        t1_getline();
        t1_putline();
    } while (!t1_prefix("cleartomark"));
    t1_length3 = pdfoffset() - pdfsaveoffset;
}

#else
boolean t1_subset(char *fontfile, char *enc, unsigned char *g)
{
    enc_file = enc;
    if (enc_file != 0)
        load_enc_file(enc_file, ext_glyph_names);
    grid = g;
    cur_file_name = fontfile;
    writet1();
}

static boolean t1_open_fontfile(char *open_name_prefix)
{
    if ((t1_file = psfopen(cur_file_name, FOPEN_RBIN_MODE)) == NULL)
        return false;
    t1_init_params(open_name_prefix);
    return true;
}

#endif

static void t1_close_font_file(char *close_name_suffix)
{
    t1_log(close_name_suffix);
    t1_close();
    cur_file_name = 0;
}

static void t1_scan_param(void) 
{
    if (!t1_scan || *t1_line != '/')
        return;
    if (t1_prefix("/lenIV")) {
        t1_lenIV = atoi(strchr(t1_line, ' ') + 1);
        return;
    }
    t1_scan_keys();
}

static void t1_eexec_init(void)
{
    int i;
    save_offset();
    t1_in_eexec = true;
    for (t1_line_ptr = t1_line, i = 0; i < 4; i++) {
        edecrypt(t1_getbyte());
        *t1_line_ptr++ = 0;
    }
    t1_putline(); /* to put the first four bytes */
}

static void t1_builtin_enc(void)
{
    int i, counter = 0;
    char *r, *p;
       /*
        * At this moment "/Encoding" is the prefix of t1_line
        * 
        * We have two possible forms of Encoding vector. The first case is
        * 
        *     /Encoding [/a /b /c...] readonly def
        * 
        * and the second case can look like
        * 
        *     /Encoding 256 array 0 1 255 {1 index exch /.notdef put} for
        *     dup 0 /x put
        *     dup 1 /y put
        *     ...
        *     readonly def
        */
    for (i = 0; i <= MAX_CHAR_CODE; i++)
        t1_builtin_glyph_names[i] = notdef;
    /* the first case */
    if (t1_prefix("/Encoding [") || t1_prefix("/Encoding[")) {
        r = strchr(t1_line, '[') + 1;
        skip(r, ' ');
        for(;;) {
            while (*r == '/') {
                for (p = t1_buf, r++;
                     *r != 32 && *r != 10 && *r != ']' && *r != '/';
                     *p++ = *r++);
                *p = 0;
                skip(r, ' ');
                if (counter > MAX_CHAR_CODE)
                    pdftex_fail("encoding vector contains more than %i names",
                            (int)(MAX_CHAR_CODE + 1));
                if (strcmp(t1_buf, notdef) != 0)
                    t1_builtin_glyph_names[counter] = xstrdup(t1_buf);
                counter++;
            }
            if (*r != 10 && *r != '%') {
                if (str_prefix(r, "] def") || str_prefix(r, "] readonly def"))
                    break;
                else {
                    remove_eol(r, t1_line);
                    pdftex_fail("a name or `] def' or `] readonly def' expected: `%s'",
                                t1_line);
                }
            }
            t1_getline();
            r = t1_line;
        }
    }
    else { /* the second case */
        do {
            t1_getline();
            for (p = t1_line; *p != 10;) {
                if (sscanf(p, "dup %u%256s put", &i, t1_buf) == 2 && 
                    *t1_buf == '/' && i >= 0 && i <= MAX_CHAR_CODE) {
                    if (strcmp(t1_buf + 1, notdef) != 0) /* don't compare the slash */
                        t1_builtin_glyph_names[i] = xstrdup(t1_buf + 1);
                    p = strstr(p, " put") + strlen(" put");
                    skip(p, ' ');
                }
                else {
                    while (*p != ' ' && *p != 10)
                        p++;
                    skip(p, ' ');
                }
            }
        } while (!t1_suffix("def"));
    }
}

#define check_subr(subr) \
    if (subr >= subr_size || subr < 0) \
        pdftex_fail("Subrs array: entry index out of range (%i)",  subr);

static void cs_store(boolean is_subr)
{
    char *p, *q;
    cs_entry *ptr;
    int subr;
    for (p = t1_line, q = t1_buf; *p != ' '; *q++ = *p++);
    *q = 0;
    if (is_subr) {
        subr = atoi(p + 1);
        check_subr(subr);
        ptr = subr_tab + subr;
        set_subr_valid(ptr);
    }
    else {
        ptr = cs_ptr++;
        if (strcmp(t1_buf + 1, notdef) == 0) /* skip the slash */
            ptr->name = notdef;
        else
            ptr->name = xstrdup(t1_buf + 1); 
    }
    ptr->used = false;
    memcpy(t1_buf, cs_start - 4, t1_cslen + 4); /* copy " RD " + cs data to t1_buf */
    for (p = cs_start + t1_cslen, q = t1_buf + t1_cslen + 4; *p != 10; *q++ = *p++);
    /* copy the end of cs data to t1_buf */
    *q++ = 10; 
    ptr->len = q - t1_buf;
    ptr->cslen = t1_cslen;
    ptr->data = xtalloc(ptr->len, byte);
    memcpy(ptr->data, t1_buf, ptr->len);
    ptr->invalid = false;
}

#define store_subr()    cs_store(true)
#define store_cs()      cs_store(false)

#define CC_STACK_SIZE       24

static integer cc_stack[CC_STACK_SIZE], *stack_ptr = cc_stack;
static cc_entry cc_tab[CS_MAX];
static boolean is_cc_init = false;


#define cc_pop(N)                       \
    if (stack_ptr - cc_stack < (N))     \
        stack_error(N);                 \
    stack_ptr -= N

#define stack_error(N) {                \
    pdftex_warn("CharString: invalid access (%i) to stack (%i entries)", \
                 N, stack_ptr - cc_stack);                               \
    cs_error = true;                    \
}

/*
static integer cc_get(integer index) 
{
    if (index <  0) {
        if (stack_ptr + index < cc_stack )
            stack_error(stack_ptr - cc_stack + index);
        return *(stack_ptr + index);
    }
    else {
        if (cc_stack  + index >= stack_ptr)
            stack_error(index);
        return cc_stack[index];
    }
}
*/

#define cc_get(N)   ((N) < 0 ? *(stack_ptr + (N)) : *(cc_stack + (N)))

#define cc_push(V)  *stack_ptr++ = V
#define cc_clear()  stack_ptr = cc_stack
    
#define set_cc(N, B, A, C) \
    cc_tab[N].nargs = A;   \
    cc_tab[N].bottom = B;  \
    cc_tab[N].clear = C;   \
    cc_tab[N].valid = true
    
static void cc_init(void)
{
    int i;
    if (is_cc_init)
        return;
    for (i = 0; i < CS_MAX; i++)
        cc_tab[i].valid = false;
    set_cc(CS_HSTEM,           true,   2, true);
    set_cc(CS_VSTEM,           true,   2, true);
    set_cc(CS_VMOVETO,         true,   1, true);
    set_cc(CS_RLINETO,         true,   2, true);
    set_cc(CS_HLINETO,         true,   1, true);
    set_cc(CS_VLINETO,         true,   1, true);
    set_cc(CS_RRCURVETO,       true,   6, true);
    set_cc(CS_CLOSEPATH,       false,  0, true);
    set_cc(CS_CALLSUBR,        false,  1, false);
    set_cc(CS_RETURN,          false,  0, false);
    /*
    set_cc(CS_ESCAPE,          false,  0, false);
    */
    set_cc(CS_HSBW,            true,   2, true);
    set_cc(CS_ENDCHAR,         false,  0, true);
    set_cc(CS_RMOVETO,         true,   2, true);
    set_cc(CS_HMOVETO,         true,   1, true);
    set_cc(CS_VHCURVETO,       true,   4, true);
    set_cc(CS_HVCURVETO,       true,   4, true);
    set_cc(CS_DOTSECTION,      false,  0, true);
    set_cc(CS_VSTEM3,          true,   6, true);
    set_cc(CS_HSTEM3,          true,   6, true);
    set_cc(CS_SEAC,            true,   5, true);
    set_cc(CS_SBW,             true,   4, true);
    set_cc(CS_DIV,             false,  2, false);
    set_cc(CS_CALLOTHERSUBR,   false,  0, false);
    set_cc(CS_POP,             false,  0, false);
    set_cc(CS_SETCURRENTPOINT, true,   2, true);
    is_cc_init = true;
}

#define cs_getchar()    cdecrypt(*data++, &cr)

#define mark_subr(n)    cs_mark(0, n)
#define mark_cs(s)      cs_mark(s, 0)

static void cs_mark(char *cs_name, int subr)
{
    byte *data;
    int i, b, cs_len;
    integer a, a1, a2;
    unsigned short cr;
    boolean cs_error = false;
    static integer OtherSubr3_last_arg = 3; /* the argument of last call to 
                                               OtherSubrs[3] */
    cs_entry *ptr;
    cc_entry *cc;
    if (cs_name == 0) {
        check_subr(subr);
        ptr = subr_tab + subr;
        if (!is_subr_valid(ptr))
            pdftex_fail("Subrs: entry `%i' is not valid", subr);
    }
    else {
        if (cs_notdef != 0 && 
            (cs_name == notdef || strcmp(cs_name, notdef) == 0))
            ptr = cs_notdef;
        else {
            for (ptr = cs_tab; ptr < cs_ptr; ptr++)
                if (strcmp(ptr->name, cs_name) == 0)
                    break;
            if (ptr == cs_ptr) {
                pdftex_warn("glyph `%s' undefined", cs_name);
                return;
            }
            if (ptr->name == notdef)
                cs_notdef = ptr;
        }
    }
    /* only marked CharString entries and invalid entries can be skipped;
       valid marked subrs must be parsed to keep the stack */
    if ((ptr->used && cs_name != 0) || ptr->invalid)
        return; 
    ptr->used = true;
    cr = 4330; 
    cs_len = ptr->cslen;
    data = ptr->data + 4;
    for (i = 0; i < t1_lenIV; i++, cs_len--)
        cs_getchar();
    while (cs_len > 0) {
        --cs_len;
        b = cs_getchar();
        if (b >= 32) {
            if (b <= 246)
                a = b - 139;
            else if (b <= 250) {
                --cs_len;
                a = ((b - 247) << 8) + 108 + cs_getchar();
            } 
            else if (b <= 254) {
                --cs_len;
                a = -((b - 251) << 8) - 108 - cs_getchar();
            } 
            else {
                cs_len -= 4;
                a =  (cs_getchar() & 0xff) << 24;
                a |= (cs_getchar() & 0xff) << 16;
                a |= (cs_getchar() & 0xff) <<  8;
                a |= (cs_getchar() & 0xff) <<  0;
                if (sizeof(integer) > 4 && (a & 0x80000000))
                    a |= ~0x7FFFFFFF;
            }
            cc_push(a);
        }
        else {
            if (b == CS_ESCAPE) {
                b = cs_getchar() + CS_1BYTE_MAX;
                cs_len--;
            }
            if (b >= CS_MAX) {
                pdftex_warn("CharString: command value out of range: %i", 
                            (int)b);
                cs_error = true;
            }
            cc = cc_tab + b;
            if (!cc->valid) {
                pdftex_warn("CharString: command not valid: %i", (int)b);
                cs_error = true;
            }
            if (cc->bottom && stack_ptr - cc_stack < cc->nargs) {
                pdftex_fail("CharString: less arguments on stack (%i) than required (%i)",
                            (int)(stack_ptr - cc_stack), (int)cc->nargs);
                cs_error = true;
            }
            if (cc->clear && cc->bottom && stack_ptr - cc_stack > cc->nargs) {
                pdftex_warn("CharString: more operands on stack (%i) than required (%i)",
                            (int)(stack_ptr - cc_stack), (int)cc->nargs);
                cs_error = true;
            }
            switch (cc - cc_tab) {
            case CS_CALLSUBR:
                a1 = cc_get(-1);
                cc_pop(1);
                mark_subr(a1);
                break;
            case CS_DIV:
                cc_pop(2);
                cc_push(0);
                break;
            case CS_CALLOTHERSUBR:
                if (cc_get(-1) == 3)
                    OtherSubr3_last_arg = cc_get(-3);
                a1 = cc_get(-2) + 2;
                cc_pop(a1);
                break;
            case CS_POP:
                cc_push(OtherSubr3_last_arg);
                /* the only case when we care about the value being pushed onto
                   stack is when POP follows CALLOTHERSUBR (changing hints by
                   OtherSubrs[3]) 
                 */
                break;
            case CS_SEAC:
                a1 = cc_get(3);
                a2 = cc_get(4);
                cc_clear();
                mark_cs(standard_glyph_names[a1]);
                mark_cs(standard_glyph_names[a2]);
                break;
            default:
                if (cc->clear)
                    cc_clear();
            }
        }
        if (cs_error) { /* an error occured during parsing */
            ptr->invalid = true;
            return;
        }
    }
}

static void t1_check_predef_enc(void)
{
    if (t1_suffix("def")) { /* predefined encoding */
        sscanf(t1_line + strlen("/Encoding"), "%256s", t1_buf);
        if (strcmp(t1_buf, "StandardEncoding") == 0)
            t1_encoding = ENC_STANDARD;
        else 
            pdftex_fail("cannot subset font (unknown predefined encoding `%s')", 
                        t1_buf);
    }
    else
        t1_encoding = ENC_BUILTIN;
}

static void t1_subset_ascii_part(void)
{
    int i;
    save_offset();
    t1_getline();
    while (!t1_prefix("/Encoding")) {
        t1_scan_param();
        t1_putline();
        t1_getline();
    }
    t1_check_predef_enc(); /* set t1_encoding to ENC_STANDARD or ENC_BUILTIN */
    if (!is_reencoded(fm_cur) && t1_encoding == ENC_STANDARD) {
        t1_putline(); /* write the predefined encoding */
        t1_glyph_names = standard_glyph_names;
        goto after_encoding;
    }
    if (t1_encoding == ENC_BUILTIN) {
        if (is_reencoded(fm_cur)) {
            while (!t1_suffix("def"))
                t1_getline(); /* skip the built-in encoding */
        }
        else
            t1_builtin_enc(); /* read the built-in encoding into
                                 t1_builtin_glyph_names */
    }
    if (is_reencoded(fm_cur))
        t1_glyph_names = external_enc();
    else /* t1_encoding == ENC_BUILTIN */
        t1_glyph_names = t1_builtin_glyph_names;
    update_cur_enc(); /* set glyph names for chars that have been moved from
                         0..32 range */
    t1_mark_extra_glyphs(); /* mark glyphs from charset */
    t1_printf("/Encoding 256 array 0 1 255 {1 index exch /.notdef put} for\n");
    for (i = 0; i <= MAX_CHAR_CODE; i++)
        if (is_used_char(i))
            t1_printf("dup %i /%s put\n" AND (int)i AND t1_glyph_names[i]);
    t1_printf("readonly def\n");
after_encoding:
    do {
        t1_getline();
        t1_scan_param();
        t1_putline();
    } while (!t1_prefix("currentfile eexec"));
    get_length1();
}

#define t1_subset_flush_subr()  t1_subset_flush(true)
#define t1_subset_flush_cs()    t1_subset_flush(false)

static void t1_subset_flush(boolean);

static void t1_subset_subrs(void)
{
    int i;
    char *s;
    cs_entry *ptr;
    t1_getline();
    while (!(t1_charstrings() || t1_subrs())) {
        t1_scan_param();
        t1_putline();
        t1_getline();
    }
    t1_cs = true;
    t1_scan = false;
    if (!t1_subrs())
       return;
    subr_size_pos = strlen("/Subrs") + 1; 
    /* subr_size_pos points to the number indicating dict size after "/Subrs" */
    subr_size = atoi(t1_line + subr_size_pos);
    subr_tab = xtalloc(subr_size, cs_entry);
    for (ptr = subr_tab; ptr - subr_tab < subr_size; ptr++) {
        ptr->data = 0;
        set_subr_invalid(ptr);
    }
    subr_array_start = xstrdup(t1_line);
    t1_getline();
    while (t1_cslen) {
        store_subr();
        t1_getline();
    }
    /* mark the first four entries without parsing */
    for (i = 0; i < subr_size && i < 4; i++)
        subr_tab[i].used = true;
    /* the end of the Subrs array might have more than one line so we need to
       concatnate them to subr_array_end. Unfortunately some fonts don't have
       the Subrs array followed by the CharStrings dict immediately. If we
       cannot find CharStrings in next 5 lines then we will not subset the
       Subrs array at all.
     */
    s = t1_buf;
    *t1_buf = 0;
    for (i = 0; i < 5; i++) {
        if (t1_charstrings())
               break;
        check_buf((t1_line_ptr - t1_line) + (s - t1_buf), T1_BUF_SIZE);
        strcat(t1_buf, t1_line);
        s += t1_line_ptr - t1_line;
        t1_getline();
    }
    subr_array_end = xstrdup(t1_buf);
    if (i == 5) { /* CharStrings not found; flush the Subrs array immediately */
        for (ptr = subr_tab; ptr - subr_tab < subr_size; ptr++)
            if (is_subr_valid(ptr))
                ptr->used = true;
        subr_max = subr_size - 1;
        strcpy(t1_buf, t1_line); /* save the current line to t1_buf */
        t1_subset_flush_subr();
        strcpy(t1_line, t1_buf); /* restore the last line from t1_buf */
        t1_line_ptr = eol(t1_line);
        t1_putline();
        t1_cs = false;
        subr_tab = 0;
        subr_array_start = subr_array_end = 0;
        t1_subset_subrs();
    }
}

static void t1_subset_flush(boolean is_subr)
{
    char *p;
    cs_entry *tab, *end_tab, *ptr;
    char *start_line, *line_end;
    int count, size_pos;
    if (is_subr) {
        start_line = subr_array_start;
        line_end = subr_array_end;
        size_pos = subr_size_pos;
        tab = subr_tab;
        end_tab = subr_tab + subr_size;
        count = subr_max + 1;
    }
    else {
        start_line = cs_dict_start;
        line_end = cs_dict_end;
        size_pos = cs_size_pos;
        tab =  cs_tab;
        end_tab = cs_ptr;
        count = cs_count;
    }
    t1_line_ptr = t1_line;
    for (p = start_line; p - start_line < size_pos;)
        *t1_line_ptr++ = *p++;
    while (isdigit(*p))
        p++;
    sprintf(t1_line_ptr, "%u", count);
    strcat(t1_line_ptr, p);
    t1_line_ptr = eol(t1_line);
    t1_putline();
    for (ptr = tab; ptr < end_tab; ptr++) {
        if (ptr->used) {
            if (is_subr)
                sprintf(t1_line, "dup %u %u", ptr - tab, ptr->cslen);
            else
                sprintf(t1_line, "/%s %u", ptr->name, ptr->cslen);
            p = strend(t1_line);
            memcpy(p, ptr->data, ptr->len);
            t1_line_ptr = p + ptr->len;
            t1_putline();
        }
        xfree(ptr->data);
        if (ptr->name != 0 && ptr->name != notdef)
            xfree(ptr->name);
    }
    sprintf(t1_line, "%s", line_end);
    t1_line_ptr = eol(t1_line);
    t1_putline();
    xfree(tab);
    xfree(start_line);
    xfree(line_end);
}

static void t1_mark_glyphs(void)
{
    int i;
    extra_glyphs_entry *p;
    mark_cs(notdef);
    for (i = 0; i <= MAX_CHAR_CODE; i++)
        if (is_used_char(i)) {
            if (t1_glyph_names[i] == notdef)
                pdftex_warn("character %i is mapped to %s", i, notdef);
            else
                mark_cs(t1_glyph_names[i]);
        }
    for (p = extra_glyphs_tab; p < extra_glyphs_ptr; p++)
        mark_cs(*p);
    xfree(extra_glyphs_tab);
}

static void t1_subset_charstrings(void)
{
    int i;
    cs_entry *ptr;
    cs_size_pos = strstr(t1_line, "/CharStrings") + strlen("/CharStrings") 
                  - t1_line + 1; 
    /* cs_size_pos points to the number indicating
       dict size after "/CharStrings" */
    cs_size = atoi(t1_line + cs_size_pos);
    cs_ptr = cs_tab = xtalloc(cs_size, cs_entry);
    cs_notdef = 0;
    cs_dict_start = xstrdup(t1_line);
    t1_getline();
    while (t1_cslen) {
        store_cs();
        t1_getline();
    }
    cs_dict_end = xstrdup(t1_line);
    t1_mark_glyphs();
    if (subr_tab != 0) {
        for (subr_max = -1, ptr = subr_tab; ptr - subr_tab < subr_size; ptr++)
            if (ptr->used && ptr - subr_tab > subr_max)
                subr_max = ptr - subr_tab;
        t1_subset_flush_subr();
    }
    for (cs_count = 0, ptr = cs_tab; ptr < cs_ptr; ptr++)
        if (ptr->used)
            cs_count++;
    t1_subset_flush_cs();
}

static void t1_subset_end(void)
{
    int k;
    while (!t1_suffix("mark currentfile closefile")) {
        t1_getline();
        t1_putline();
    }
    get_length2();
    save_offset();
    t1_in_eexec = false;
    t1_cs = false;
    while (!t1_prefix("cleartomark")) {
        t1_getline();
        t1_putline();
    }
    get_length3();
}

void writet1(void)
{
    read_encoding_only = false;
    if (!is_included(fm_cur)) { /* scan parameters from font file */
        if (!t1_open_fontfile("{"))
            return;
        t1_scan_only();
        t1_close_font_file("}");
        return;
    }
    if (!is_subsetted(fm_cur)) { /* include entire font */
        if (!t1_open_fontfile("<<"))
            return;
        t1_include();
        t1_close_font_file(">>");
        return;
    } 
    /* partial downloading */
    if (!t1_open_fontfile("<"))
        return;
    t1_subset_ascii_part();
    t1_eexec_init();
    cc_init();
    cs_tab = 0;
    cs_dict_start =  cs_dict_end = 0;
    subr_tab = 0;
    subr_array_start = subr_array_end = 0;
    t1_subset_subrs();
    t1_subset_charstrings();
    t1_subset_end();
    t1_close_font_file(">");
}
