#include <stdlib.h>
#include <stddef.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <GString.h>
#include <gmem.h>
#include <gfile.h>
#include <config.h>
#include "Object.h"
#include "Stream.h"
#include "Array.h"
#include "Dict.h"
#include "XRef.h"
#include "Catalog.h"
#include "Page.h"
#include "PDFDoc.h"
#include "Params.h"
#include "Error.h"

#include "epdf.h"

#define objfree(obj)  if (!obj.isNull())    obj.free()

struct IndirectObj {
    Ref ref;             // ref in original PDF
    integer newNum;      // new object number in output PDF
    IndirectObj *next;   // next entry in list of indirect objects
    GBool skip;          // to mark objects that are skipped (font file streams)
    integer fontnameNum; // object number of font name of this object (BaseFont 
                         //     in Font dict and FontName in FontDescriptor
                         //     dict)
};

GBool printCommands = gFalse;
IndirectObj *indirectObjList;
static GBool isInit = gFalse;

static int addIndirectObj(Ref r, integer fontfileObjnum, integer fontnameObjnum)
{
    IndirectObj *p, *q, *n = new IndirectObj[1];
    n->ref = r;
    n->next = 0;
    n->fontnameNum = fontnameObjnum;
    if (indirectObjList == 0)
        indirectObjList = n;
    else {
        for (p = indirectObjList; p != 0; p = p->next) {
            if (p->ref.num == r.num && p->ref.gen == r.gen) {
                delete n;
                return p->newNum;
            }
            q = p;
        }
        q->next = n;
    }
    if (fontfileObjnum == 0) {
        n->newNum = pdfnewobjnum();
        n->skip = gFalse;
    }
    else {
        n->newNum = fontfileObjnum;
        n->skip = gTrue;
    }
    return n->newNum;
}

static void copyObject(Object *, IndirectObj *);

static void copyDictEntry(Object *obj, int i)
{
    Object obj1;
    pdf_printf("/%s ", obj->dictGetKey(i));
    obj->dictGetValNF(i, &obj1);
    copyObject(&obj1, 0);
    obj1.free();
    pdf_puts("\n");
}
        
static void copyStreamDict(Object *obj)
{
    int i, l;
    char *key;
    if (!obj->isDict())
        pdftex_fail("xpdf: invalid stream dict type <%s>", obj->getTypeName());
    for (i = 0, l = obj->dictGetLength(); i < l; ++i) {
        key = obj->dictGetKey(i);
        if (strcmp("Length", key) == 0      ||
            strcmp("Filter", key) == 0      ||
            strcmp("DecodeParms", key) == 0 ||
            strcmp("F", key) == 0           ||
            strcmp("FFilter", key) == 0     ||
            strcmp("FDecodeParms", key) == 0)
            continue;
        copyDictEntry(obj, i);
    }
}

static void copyDict(Object *obj, IndirectObj *r)
{
    Object obj1;
    int i, l;
    char *key;
    if (!obj->isDict())
        pdftex_fail("xpdf: invalid dict type <%s>", obj->getTypeName());
    for (i = 0, l = obj->dictGetLength(); i < l; ++i) {
        key = obj->dictGetKey(i);
        if (r != 0 && r->fontnameNum != 0) {
            if (strcmp("BaseFont", key) == 0 || strcmp("FontName", key) == 0) {
                pdf_printf("/%s %i 0 R\n", key, r->fontnameNum);
                continue;
            }
        }
        copyDictEntry(obj, i);
    }
}

static void copyStream(Stream *str)
{
    int c;
    str->reset();
    while ((c = str->getChar()) != EOF)
        pdfout(c);
}

static void copyProcSet(Object *obj)
{
    int i, l;
    Object procset;
    if (!obj->isArray())
        pdftex_fail("xpdf: invalid ProcSet type <%s>", obj->getTypeName());
    pdf_puts("/ProcSet [ ");
    for (i = 0, l = obj->arrayGetLength(); i < l; ++i) {
        obj->arrayGet(i, &procset);
        if (!procset.isName())
            pdftex_fail("xpdf: invalid ProcSet type <%s>", procset.getTypeName());
        pdf_printf("/%s ", procset.getName());
        procset.free();
    }
    pdf_puts("]\n");
}

void copyFont(char *tag, Object *fontRef)
{
    Object fontdict, basefont, fontdescRef, fontdesc, fontfileRef, fontfile, charset;
    int texfont;
    fontdict.initNull();
    fontRef->fetch(&fontdict);
    if (!fontdict.isDict())
        pdftex_fail("xpdf: invalid font dict type <%s>", fontdict.getTypeName());
    basefont.initNull();
    fontdict.dictLookup("BaseFont", &basefont);
    if (!basefont.isName())
        pdftex_fail("xpdf: invalid base font type <%s>", basefont.getTypeName());
    texfont = lookup_fbname(basefont.getName());
    fontdescRef.initNull();
    fontdesc.initNull();
    fontfileRef.initNull();
    fontfile.initNull();
    if (texfont >= 0 && 
        fontdict.dictLookupNF("FontDescriptor", &fontdescRef) && 
        fontdescRef.isRef() && fontdescRef.fetch(&fontdesc) &&
        fontdesc.dictLookupNF("FontFile", &fontfileRef) &&
        fontfileRef.isRef() && fontfileRef.fetch(&fontfile)) {
        if (fontdesc.dictLookup("CharSet", &charset) && 
            charset.isString() && is_subsetable(texfont)) {
            mark_glyphs(texfont, charset.getString()->getCString());
            charset.free();
        }
        else
            embed_whole_font(texfont);
        // mark the font file stream not to be copied
        addIndirectObj(fontfileRef.getRef(), fontfile_objnum(texfont), 0);
        // replace the original font name in font descriptor by the name
        //     fetched by pdftex
        addIndirectObj(fontdescRef.getRef(), 0, fontname_objnum(texfont));
    }
    // similiarly replace the original font name in font object by the name
    //     fetched by pdftex
    pdf_printf("/%s %d 0 R ", tag, 
                addIndirectObj(fontRef->getRef(), 0, fontdesc.isNull() ? 0 : 
                                                     fontname_objnum(texfont)));
    objfree(fontfile);
    objfree(fontfileRef);
    objfree(fontdesc);
    objfree(fontdescRef);
    objfree(fontdict);
    objfree(basefont);
}

void copyFontResources(Object *obj)
{
    Object fontRef;
    int i, l;
    if (!obj->isDict())
        pdftex_fail("xpdf: invalid font resources type <%s>", obj->getTypeName());
    pdf_puts("/Font << ");
    for (i = 0, l = obj->dictGetLength(); i < l; ++i) {
        obj->dictGetValNF(i, &fontRef);
        if (fontRef.isRef())
            copyFont(obj->dictGetKey(i), &fontRef);
        else {
            pdf_printf("/%s ", obj->dictGetKey(i));
            copyObject(&fontRef, 0);
        }
        fontRef.free();
    }
    pdf_puts(">>\n");
}

void copyOtherResources(Object *obj, char *key)
{
    Object obj1;
    int i, l;
    if (!obj->isDict())
        pdftex_fail("xpdf: invalid other resources type <%s>", obj->getTypeName());
    pdf_printf("/%s << ", key);
    for (i = 0, l = obj->dictGetLength(); i < l; ++i) {
        obj->dictGetValNF(i, &obj1);
        if (obj1.isRef())
            pdf_printf("/%s %d 0 R ", obj->dictGetKey(i), 
                       addIndirectObj(obj1.getRef(), 0, 0));
        else 
            copyObject(&obj1, 0);
        obj1.free();
    }
    pdf_puts(">>\n");
}

void copyObject(Object *obj, IndirectObj *r)
{
    Object obj1;
    int  i, l;
    char *p;
    if (obj->isBool()) {
        pdf_printf("%s", obj->getBool() ? "true" : "false");
    }
    else if (obj->isInt()) {
        pdf_printf("%i", obj->getInt());
    }
    else if (obj->isReal()) {
        pdf_printf("%g", obj->getReal());
    }
    else if (obj->isNum()) {
        pdf_printf("%g", obj->getNum());
    }
    else if (obj->isString()) {
        pdf_puts("(");
        for (p = obj->getString()->getCString(); *p; p++)
            if (*p == '(' || *p == ')' || *p == '\\')
                pdf_printf("\\%c", *p);
            else if (!(*p > 0x20 && *p < 0x80 - 1))
                pdf_printf("\\%03o", *p);
            else
                pdfout(*p);
        pdf_puts(")");
    }
    else if (obj->isName()) {
        pdf_printf("/%s", obj->getName());
    }
    else if (obj->isNull()) {
    }
    else if (obj->isArray()) {
        pdf_puts("[");
        for (i = 0, l = obj->arrayGetLength(); i < l; ++i) {
            obj->arrayGetNF(i, &obj1);
            if (!obj1.isName())
                pdf_puts(" ");
            copyObject(&obj1, 0);
            obj1.free();
        }
        pdf_puts("]");
    }
    else if (obj->isDict()) {
        pdf_puts("<<\n");
        copyDict(obj, r);
        pdf_puts(">>");
    }
    else if (obj->isStream()) {
        obj1.initDict(obj->getStream()->getDict());
        obj->getStream()->getDict()->incRef();
        pdf_puts("<<\n");
        // copyStreamDict(&obj1);
        copyDict(&obj1, 0);
        obj1.free();
        pdf_puts(">>\n");
        pdf_puts("stream\n");
        copyStream(obj->getStream()->getBaseStream());
        pdf_puts("endstream\n");
    }
    else if (obj->isRef()) {
        pdf_printf("%d 0 R", addIndirectObj(obj->getRef(), 0, 0));
    }
    else {
        pdftex_fail("xpdf: type <%s> cannot be copied", obj->getTypeName());
    }
}

static void writeRefs()
{
    Object obj1;
    IndirectObj *r, *n;
    for (r = indirectObjList; r != 0; r = r->next) {
        if (r->skip)
            continue;
        zpdfbeginobj(r->newNum);
        xref->fetch(r->ref.num, r->ref.gen, &obj1);
        copyObject(&obj1, r);
        pdf_puts("\nendobj\n");
        /*
        if (!obj1.isStream())
            pdf_puts("\nendobj\n");
        */
        obj1.free();
    }
    for (r = indirectObjList; r != 0; r = n) {
        n = r->next;
        delete r;
    }
}

integer read_pdf_info(char *image_name, integer page_num)
{
    PDFDoc *doc;
    GString *docName;
    Page *page;
    Object contents, obj, resources;
    int n;
    // initialize
    if (!isInit) {
        initParams(xpdfConfigFile);
        errorInit();
        isInit = gTrue;
    }
    // open PDF file
    xref = 0;
    docName = new GString(image_name);
    doc = new PDFDoc(docName);
    if (!doc->isOk() || !doc->okToPrint())
        return -1;
    epdf_doc = (void *)doc;
    epdf_xref = (void *)xref;
    n = doc->getCatalog()->getNumPages();
    if (page_num <= 0 || page_num > n)
        pdftex_fail("xpdf: required page does not exists <%i>", n);
    // get the required page
    page = doc->getCatalog()->getPage(page_num);
    if (page->isCropped()) {
        epdf_width = (int)(page->getCropX2() - page->getCropX1());
        epdf_height = (int)(page->getCropY2() - page->getCropY1());
        epdf_orig_x = (int)page->getCropX1();
        epdf_orig_y = (int)page->getCropY1();
    }
    else {
        epdf_width = (int)(page->getX2() - page->getX1());
        epdf_height = (int)(page->getY2() - page->getY1());
        epdf_orig_x = (int)page->getX1();
        epdf_orig_y = (int)page->getY1();
    }
    return 0;
}

void write_epdf(integer img)
{
    Page *page;
    Object contents, obj1, obj2;
    char *key;
    int i, l;
    xref = (XRef *)epdf_xref;
    page = ((PDFDoc *)epdf_doc)->getCatalog()->getPage(epdf_selected_page);
    indirectObjList = 0;
    // write the Page header
    pdf_puts("/Type /XObject\n");
    pdf_puts("/Subtype /Form\n");
    pdf_puts("/FormType 1\n");
    pdf_puts("/Matrix [1 0 0 1 0 0]\n");
    if (page->isCropped())
        pdf_printf("/BBox [%i %i %i %i]\n",
                  (int)page->getCropX1(),
                  (int)page->getCropY1(),
                  (int)page->getCropX2(),
                  (int)page->getCropY2());
    else 
        pdf_printf("/BBox [%i %i %i %i]\n",
                  (int)page->getX1(),
                  (int)page->getY1(),
                  (int)page->getX2(),
                  (int)page->getY2());
    // write the Resources dictionary
    obj1.initDict(page->getResourceDict());
    page->getResourceDict()->incRef();
    if (!obj1.isDict())
        pdftex_fail("xpdf: invalid resources type <%s>", obj1.getTypeName());
    pdf_puts("/Resources <<\n");
    for (i = 0, l = obj1.dictGetLength(); i < l; ++i) {
        obj1.dictGetVal(i, &obj2);
        key = obj1.dictGetKey(i);
        if (strcmp("Font", key) == 0)
            copyFontResources(&obj2);
        else if (strcmp("ProcSet", key) == 0)
            copyProcSet(&obj2);
        else
            copyOtherResources(&obj2, key);
        obj2.free();
    }
    obj1.free();
    pdf_puts(">>\n");
    // write the page contents
    page->getContents(&contents);
    if (contents.isStream()) {
        obj1.initDict(contents.getStream()->getDict());
        contents.getStream()->getDict()->incRef();
        copyDict(&obj1, 0);
        pdf_puts(">>\nstream\n");
        copyStream(contents.getStream()->getBaseStream());
        pdf_puts("endstream\nendobj\n");
        obj1.free();
    }
    else if (contents.isArray()) {
        pdfbeginstream();
        for (i = 0, l = contents.arrayGetLength(); i < l; ++i) {
            contents.arrayGet(i, &obj1);
            copyStream(obj1.getStream());
            obj1.free();
        }
        pdfendstream();
    }
    else
        pdftex_fail("xpdf: invalid page contents type <%s>", contents.getTypeName());
    contents.free();
    // write out all indirect objects
    writeRefs();
}

void epdf_delete()
{
    xref = (XRef *)epdf_xref;
    delete (PDFDoc *)epdf_doc;
}

void epdf_check_mem()
{
    if (isInit) {
        freeParams();
        Object::memCheck(errFile);
        gMemReport(errFile);
    }
}
