/* -----------------------------------------------------------------------
 * This is TIFF support to pdfTeX.
 * -----------------------------------------------------------------------
 */

#ifdef HAVE_TIFF

#include "ptexlib.h"
#include "image.h"

void read_tif_info(integer img)
{
    float xres = 0, 
          yres = 0;
	uint16 res_unit = RESUNIT_NONE;
    uint16 photometric = PHOTOMETRIC_RGB;
    TIFF *tif = tif_ptr(img) = TIFFOpen(img_name(img), FOPEN_RBIN_MODE);
    if (tif == 0)
        pdftex_fail("reading TIFF image failed");
    TIFFGetField(tif, TIFFTAG_IMAGELENGTH, &img_height(img));
    TIFFGetField(tif, TIFFTAG_IMAGEWIDTH, &img_width(img));
    TIFFGetField(tif, TIFFTAG_XRESOLUTION, &xres);
    TIFFGetField(tif, TIFFTAG_YRESOLUTION, &yres);
    TIFFGetField(tif, TIFFTAG_RESOLUTIONUNIT, &res_unit);
    TIFFGetField(tif, TIFFTAG_PHOTOMETRIC, &photometric);
    switch (res_unit) {
    case RESUNIT_INCH: /* pixels per inch */
        break;
    case RESUNIT_CENTIMETER: /* pixels per cm */
        xres *= 2.54;
        yres *= 2.54;
        break;
    case RESUNIT_NONE: /* no resolution */
        xres = 0;
        yres = 0; 
        break;
    }
    img_xres(img) = xres + 0.5;
    img_yres(img) = yres + 0.5;
    switch (photometric) {
    case PHOTOMETRIC_MINISWHITE:
    case PHOTOMETRIC_MINISBLACK:
    case PHOTOMETRIC_MASK:
        img_color(img) = IMAGE_COLOR_B;
        break;
    default:
    /*
    case PHOTOMETRIC_RGB:
    case PHOTOMETRIC_PALETTE:
    case PHOTOMETRIC_SEPARATED:
    case PHOTOMETRIC_YCBCR:
    case PHOTOMETRIC_CIELAB:
    case PHOTOMETRIC_LOGL:
    case PHOTOMETRIC_LOGLUV:
    */
        img_color(img) = IMAGE_COLOR_C;
    }
}

void write_tif(integer img)
{
    TIFF *tif = tif_ptr(img);
    int width = img_width(img),
        height = img_height(img),
        w, h;
    uint16 bitspersample = 8;
    uint32 *raster = (uint32 *) _TIFFmalloc(width * height* sizeof(uint32)),
           *r = raster;
    unsigned char a, b, c, mask;
    if (raster == 0)
        pdftex_fail("cannot allocate memory for reading image");
    pdf_printf("/Type /XObject\n/Subtype /Image\n/Width %i\n/Height %i\n",
                width, height );
    if (img_color(img) == IMAGE_COLOR_B) {
        TIFFGetField(tif, TIFFTAG_BITSPERSAMPLE, &bitspersample);
        switch (bitspersample) {
        case 1:
        case 2:
            break;
        case 3:
        case 4:
            bitspersample = 4;
            break;
        default:
            bitspersample = 8;
        }
        pdf_printf("/ColorSpace /DeviceGray\n/BitsPerComponent %i\n", 
                   (int)bitspersample);
    }
    else
        pdf_puts("/ColorSpace /DeviceRGB\n/BitsPerComponent 8\n");
    if (!TIFFReadRGBAImage(tif, width, height, raster, 0))
        pdftex_fail("reading TIFF image failed");
    pdfbeginstream();
    if (img_color(img) == IMAGE_COLOR_C) {
        for (h = 0; h < height; h++) {
            r = raster + width*(height - h - 1);
            for (w = 0; w < width; w++, r++) {
                pdfroom(3);
                pdfbuf[pdfptr++] = TIFFGetR(*r);
                pdfbuf[pdfptr++] = TIFFGetG(*r);
                pdfbuf[pdfptr++] = TIFFGetB(*r);
            }
        }
    }
    else {
        if (bitspersample == 8) {
            for (h = 0; h < height; h++) {
                r = raster + width*(height - h - 1);
                for (w = 0; w < width; w++, r++) {
                    a = TIFFGetR(*r);
                    a = TIFFGetG(*r);
                    a = TIFFGetB(*r);
                    pdfout(a);
                }
            }
        }
        else {
            mask = (1 << bitspersample) - 1;
            for (h = 0; h < height; h++) {
                r = raster + width*(height - h - 1);
                b = c = 0;
                for (w = 0; w < width; w++, r++) {
                    a = TIFFGetR(*r);
                    a = TIFFGetG(*r);
                    a = TIFFGetB(*r);
                    b = (b << bitspersample) | (a & mask);
                    c += bitspersample;
                    if (c == 8) {
                        pdfout(b);
                        b = c = 0;
                    }
                }
                if (c != 0)
                    pdfout(b);
            }
        }
    }
    pdfendstream();
}
#endif
