/*  t1accent -- adds the new accents into Type1 fonts
    Copyright (C) 1998 Petr Olsak, <olsak@math.feld.cvut.cz>
    Version April 1998 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <limits.h>

#ifndef max
#define max(a,b)        (((a) > (b)) ? (a) : (b))
#define min(a,b)        (((a) < (b)) ? (a) : (b))
#endif

#define MAXGBUF 30000   /* the size of array for all CharStrings */
#define MAXIMEM  8000   /* the work space for decrypted CharString */
#define MAXBUF 200      /* limit of the work buffer */
#define MAXSTACK 25     /* maximum of Type1 stack, sufficient */
#define MAXSUBS 300     /* maximum number of subroutines */
#define MAXINFILES 200  /* maximum number of processed pfb files */
#define MAXVARS 200     /* maximum number of VAR declared in conf file */
#define MAXNEWCHARS 200 /* maximum number of CHAR declared in conf file */
#define MAXREMARKS 20   /* maximum number of REM lines */
#define MAXCHARS 300    /* maximum number of chars in input pfb file */

/* main string memory will be allocated via malloc:
   static char gbuf[MAXGBUF];
*/
unsigned char *pbuf, *gbuf, *fixpbuf;
unsigned char *csname[50];   /* the array of control sequences */
int cscode[50];
int last_csname, start_type1code;
char buf[MAXBUF];   /* temporary buffer */
char fname[MAXBUF];
char RDcommand[3], NDcommand[3], NPcommand[3];

char pattern[MAXBUF];
int pip, ispattern;
int lenIV;

int numfiles=0;
unsigned char *infile[MAXINFILES];

int imem[MAXIMEM];  /* the decrypted Chartrings */
unsigned int pim=1, fixpim, topim;

int numrems = 0;
unsigned char *remark[MAXREMARKS];
int srcisread = 0, nameisread = 0;
unsigned char *srcdir, *newname, *oriname;
int diffname;

int numsubs=0, pfbnumsubs, pfbchars;
struct subrtype {      /* structure for new subroutine */
  unsigned int code;
  unsigned char *name;
  int vvaxis, lx, ly, ux, uy ;
  } subr[MAXSUBS];

int numvars=0;
struct varstruct {     /* structure for variable declared by VAR */
  unsigned char *name;
  float value;
} variable[MAXVARS];

int numchars=0;
int sumchars;
struct newcharstruct { /* structure for new char declared by CHAR */
  unsigned char *name, *origname, *origcode;
  unsigned int owncode;
  int origlen, nsub, x, y, dsbx, dwx, dy, type;
} newchar[MAXNEWCHARS];

struct savedsubrstruct { /* structure for /Subrs from pfb */
  unsigned char *code;
  int len;
} savedsubr[MAXSUBS];

struct savedcharstruct { /* structure for /CharStrings from pfb */
  unsigned char *code, *name;
  int len;
} savedchars[MAXCHARS];

unsigned char *encvector[256];  /* encoding vector */
int encisread=0;

int prst = 0;
float rs[MAXSTACK];      /* stack with real numbers */

float scaleX, scaleY, curX, curY;     /* global variables */
int cX, cY, lX, lY, uX, uY, inpath, pX, pY, vvax, glX, glY, guX, guY;

int line=0;

FILE *fin, *fenc, *fpfb, *fpps;

#define VALUE 0
#define HSTEM 1
#define VSTEM 3
#define VMOVETO 4
#define RLINETO 5
#define HLINETO 6
#define VLINETO 7
#define RRCURVETO 8
#define CLOSEPATH 9
#define CALLSUBR 10
#define RETURN 11
#define ESCAPE 12
#define HSBW 13
#define ENDCHAR 14
#define RMOVETO 21
#define HMOVETO 22
#define VHCURVETO 30
#define HVCURVETO 31
#define DOTSECTION 1200
#define VSTEM3 1201
#define HSTEM3 1202
#define SEAC 1206
#define SBW 1207
#define DIV 1212
#define CALLOTHERSUBR 1216
#define POP 1217
#define SETCURRENTPOINT 1233
#define MOVETO   1301
#define LINETO   1302
#define CURVETO  1303
#define RCURVETO 1304
#define SCALED   1310
#define VVAXIS   1311
#define BGROUP   1401
#define EGROUP   1402
#define NUMSUB   1403
#define PUTORIGIN  1501
#define PUTSIDEBAR 1522
#define PUTCENTER  1503
#define PUTAFTER  1504
#define UNKNOWN  1599
#define MAGIC    31243
#define BEGIN  1
#define CLOSE  2
#define CURXY  3
#define CURX   4
#define CURY   5
#define CURXY6 6
#define CURHV4 7
#define CURVH4 8
#define SAVECHAR 1
#define SAVESUBR 2

void error (char* format, char* value)
{
/* prints an error message and exists
*/
  if (line>0) fprintf(stderr,"\nError: (line %d) -- ", line);
  else fprintf(stderr,"\nError: -- ");
  if (value) fprintf(stderr, format, value);
  else fprintf(stderr, format);
  fprintf(stderr,"\n");
  exit(2);
}

unsigned char* add_gbuf(unsigned char* p)
{
/* stores a new string in global buffer
*/
  unsigned char *pp;
  pp=pbuf;
  pbuf += strlen(p)+1;
  if (pbuf >= &gbuf[MAXGBUF])
     error ("MAXGBUF (global string memory) overflows",NULL);
  strcpy(pp, p);
  return pp;
}

void store_gbuf(int c)
{
/* stores a new byte in global buffer
*/
  if (pbuf >= &gbuf[MAXGBUF])
     error ("MAXGBUF (global string memory) overflows",NULL);
  pbuf[0] = c;
  pbuf++;
}

void init_csnames(void)
{
/* initialises standard operator names in global buffer
*/
  int i=0;

  gbuf = pbuf = (unsigned char*) malloc(MAXGBUF);
  if (!gbuf) error ("cannot allocated main strig memory", NULL);

  csname[i] = add_gbuf("{");               cscode[i++] = BGROUP;
  csname[i] = add_gbuf("}");               cscode[i++] = EGROUP;
  start_type1code = i;
  csname[i] = add_gbuf("hstem");           cscode[i++] = HSTEM;
  csname[i] = add_gbuf("vstem");           cscode[i++] = VSTEM;
  csname[i] = add_gbuf("vmoveto");         cscode[i++] = VMOVETO;
  csname[i] = add_gbuf("rlineto");         cscode[i++] = RLINETO;
  csname[i] = add_gbuf("hlineto");         cscode[i++] = HLINETO;
  csname[i] = add_gbuf("vlineto");         cscode[i++] = VLINETO;
  csname[i] = add_gbuf("rrcurveto");       cscode[i++] = RRCURVETO;
  csname[i] = add_gbuf("closepath");       cscode[i++] = CLOSEPATH;
  csname[i] = add_gbuf("callsubr");        cscode[i++] = CALLSUBR;
  csname[i] = add_gbuf("return");          cscode[i++] = RETURN;
  csname[i] = add_gbuf("escape");          cscode[i++] = ESCAPE;
  csname[i] = add_gbuf("hsbw");            cscode[i++] = HSBW;
  csname[i] = add_gbuf("endchar");         cscode[i++] = ENDCHAR;
  csname[i] = add_gbuf("rmoveto");         cscode[i++] = RMOVETO;
  csname[i] = add_gbuf("hmoveto");         cscode[i++] = HMOVETO;
  csname[i] = add_gbuf("vhcurveto");       cscode[i++] = VHCURVETO;
  csname[i] = add_gbuf("hvcurveto");       cscode[i++] = HVCURVETO;
  csname[i] = add_gbuf("dotsection");      cscode[i++] = DOTSECTION;
  csname[i] = add_gbuf("vstem3");          cscode[i++] = VSTEM3;
  csname[i] = add_gbuf("hstem3");          cscode[i++] = HSTEM3;
  csname[i] = add_gbuf("seac");            cscode[i++] = SEAC;
  csname[i] = add_gbuf("sbw");             cscode[i++] = SBW;
  csname[i] = add_gbuf("div");             cscode[i++] = DIV;
  csname[i] = add_gbuf("callothersubr");   cscode[i++] = CALLOTHERSUBR;
  csname[i] = add_gbuf("pop");             cscode[i++] = POP;
  csname[i] = add_gbuf("setcurrentpoint"); cscode[i++] = SETCURRENTPOINT;
  csname[i] = add_gbuf("moveto");          cscode[i++] = MOVETO;
  csname[i] = add_gbuf("lineto");          cscode[i++] = LINETO;
  csname[i] = add_gbuf("curveto");         cscode[i++] = CURVETO;
  csname[i] = add_gbuf("rcurveto");        cscode[i++] = RCURVETO;
  csname[i] = add_gbuf("scaled");          cscode[i++] = SCALED;
  csname[i] = add_gbuf("vvaxis");          cscode[i++] = VVAXIS;
  last_csname=i-1;
  buf[MAXBUF-1] = 0;
}

/* The decrypting algorithm from pfb is token over the t1disasm from t1utils
   by I. Lee Hetherington */

int immediate_eexec;
/* int32 must be at least 32-bit and uint16 must be at least 16-bit */
#if INT_MAX >= 0x7FFFFFFFUL
typedef int int32;
#else
typedef long int32;
#endif
#if USHRT_MAX >= 0xFFFFUL
typedef unsigned short uint16;
#else
typedef unsigned int uint16;
#endif
typedef unsigned char byte;
static uint16 er, cr;
static uint16 c1 = 52845, c2 = 22719;
int first_byte;
int is_pfb;
int32 pfb_remaining;
int in_eexec;
int savedchar, is_textspace;
#define cgetc()  cdecrypt((byte) (egetc() & 0xff))

static int bgetc()
{
/* returns a single character at a time from a PFB file.
   This stream is mixed ASCII and binary bytes.  The section
   headers are removed, and \r is replaced by \n in ASCII sections.
*/
  int c;

  c = fgetc(fpfb);

  if (c == EOF) {
    fprintf(stderr,"remaining = %d", pfb_remaining);
    return EOF;
  }
  if (first_byte) {
    /* Determine if this is a PFB file by looking at first byte. */
    if (c != 0x80) error ("the file %s is not in pfb format", fname);
    first_byte = 0;
  }
  /* PFB */
  if (pfb_remaining == 0) {
    /* beginning of block---we know c == 0x80 at this point */
    c = fgetc(fpfb);
    switch (c) {
    case 1:
      is_pfb = 1;
      break;
    case 2:
      is_pfb = 2;
      break;
    case 3:
      return EOF;
    default: error("the file %s is not in PFB format", fname);
    }
    /* get block length */
    pfb_remaining = (int32) (fgetc(fpfb) & 0xff);
    pfb_remaining |= (int32) (fgetc(fpfb) & 0xff) << 8;
    pfb_remaining |= (int32) (fgetc(fpfb) & 0xff) << 16;
    pfb_remaining |= (int32) (fgetc(fpfb) & 0xff) << 24;
    /* get character */
    c = fgetc(fpfb);
    if (c == EOF)
      return EOF;
  }
  --pfb_remaining;
  /* in ASCII section change return to newline */
  if (is_pfb == 1 && c == '\r') {
    c = fgetc(fpfb);
    if (c != '\n') {
      ungetc(c, fpfb);
      c = '\n' ;
    } else --pfb_remaining;
  }
  return c;
}

static byte edecrypt(byte cipher)
{
/* an eexec decryption
*/
  byte plain;

  plain = (byte) (cipher ^ (er >> 8));
  er = (uint16) ((cipher + er) * c1 + c2);
  return plain;
}

static byte cdecrypt(byte cipher)
{
/* a CharString decryption
*/
  byte plain;

  plain = (byte) (cipher ^ (cr >> 8));
  cr = (uint16) ((cipher + cr) * c1 + c2);
  return plain;
}

static int egetc()
{
/* returns a single byte at a time through (possible) eexec
   decryption.  When immediate_eexec is 1 it fires up the eexec decryption
   machinery.
*/
  int c;

  if (savedchar >= 0) {
    c = savedchar;
    savedchar = -1;
    return c;
  }
  if ((c = bgetc()) == EOF) return EOF;

  if (!in_eexec) {
    if (immediate_eexec) {
      /* start eexec decryption */
      in_eexec = 1;
      er = 55665;
      /* toss out four random bytes */
      (void) edecrypt((byte) (bgetc() & 0xff));
      (void) edecrypt((byte) (bgetc() & 0xff));
      (void) edecrypt((byte) (bgetc() & 0xff));
      (void) edecrypt((byte) (bgetc() & 0xff));
    }
    return c;
  } else {
    c = edecrypt((byte) (c & 0xff));
    if (is_textspace && c == '\r') {
      if ((c = bgetc()) == EOF) return EOF;
      c = edecrypt((byte) (c & 0xff));
      if (c!='\n') {
        savedchar = c;
        c = '\n';
      }
    }
    return c;
  }
}

int Fgetc (FILE *fh)
{
/* This is a low level input filter for config file. It transforms
   white space to space and ignores a % comment
*/
  int c;

restart:
  c = fgetc(fh);
  if (c=='\n') line++;
  if (c=='\t' || c=='\r' || c=='\n' || c==0) c = ' ';
  if (c=='%') {
    while (c!='\n' && c!=EOF) c = fgetc(fh);
    line++;
    if (c=='\n') goto restart;
  }
  return c;
}

void store_stack(float val)
{
/* stores one value to rs[] stack
*/
  if (prst>=MAXSTACK) error ("stack overflow", NULL);
  rs[prst++] = val;
}

int read_token(FILE *fh, int status)
{
/* reads one token (word or number separated by white space)
   from input stream. If status then function returns the number of
   control sequence. The value stores into stack.
*/
  int c, p;

  c = ' ';
  buf[0] = 0;
  while (c==' ') c = Fgetc(fh);
  if (c==EOF) return EOF;
  p = 0;
  while (c!=' ' && c!=EOF) {
    if (p>MAXBUF-2) error("control sequence '%s' too long", buf);
    buf[p++] = c;
    c = Fgetc(fh);
  }
  buf[p] = 0;

  if (status == 0) return 0;
  
  p = 0;
  if (buf[0] == '-') p = 1;
  if (isdigit(buf[p]) || (buf[p]=='.' && isdigit(buf[p+1]))) {
    store_stack (atof(buf));
    return VALUE;
  }
  for (p=0;p<numvars;p++)
    if (strcmp(buf,variable[p].name)==0) {
      store_stack (variable[p].value);
      return VALUE;
    }
  for (p=0;p<=last_csname;p++)
    if (strcmp(buf,csname[p])==0)  return cscode[p];
   return UNKNOWN;
}

void save_code(int num, int code)
{
/* stores a decrypted part of code to imem[] in the form
   <num> operands from rs[] stack plus one operator declared by 'code'.
   save_code() does an optimalisation of code (rlineto -> hlineto) too.
*/
  int i, x, y;
  if (MAXIMEM-pim<num+2) error ("MAXIMEM exceeded", NULL);
  if (num > prst) num = prst;
  prst -= num;
  switch (code) {                      /* optimalisation */
    case RMOVETO: x = rs[prst]; y = rs[prst+1];
                  if (x==0 && y==0) break;
                  if (x==0) {
                    imem[pim++] = y;
                    imem[pim++] = MAGIC; imem[pim++] = VMOVETO;
                  }
                  if (y==0) {
                    imem[pim++] = x;
                    imem[pim++] = MAGIC; imem[pim++] = HMOVETO;
                  }
                  if (x!=0 && y!=0) {
                    imem[pim++] = x; imem[pim++] = y;
                    imem[pim++] = MAGIC; imem[pim++] = RMOVETO;
                  }
                  break;
    case RLINETO: x = rs[prst]; y = rs[prst+1];
                  if (x==0 && y==0) break;
                  if (x==0) {
                    imem[pim++] = y;
                    imem[pim++] = MAGIC; imem[pim++] = VLINETO;
                  }
                  if (y==0) {
                    imem[pim++] = x;
                    imem[pim++] = MAGIC; imem[pim++] = HLINETO;
                  }
                  if (x!=0 && y!=0) {
                    imem[pim++] = x; imem[pim++] = y;
                    imem[pim++] = MAGIC; imem[pim++] = RLINETO;
                  }
                  break;
    case RRCURVETO: x = rs[prst]; y = rs[prst+5];
                  if (x==0 && y==0) {
                    imem[pim++] = rs[prst+1];  imem[pim++] = rs[prst+2];
                    imem[pim++] = rs[prst+3];  imem[pim++] = rs[prst+4];
                    imem[pim++] = MAGIC; imem[pim++] = VHCURVETO;
                    break; }
                  y = rs[prst+1]; x = rs[prst+4];
                  if (x==0 && y==0) {
                    imem[pim++] = rs[prst];    imem[pim++] = rs[prst+2];
                    imem[pim++] = rs[prst+3];  imem[pim++] = rs[prst+5];
                    imem[pim++] = MAGIC; imem[pim++] = HVCURVETO;
                    break; }
    default: for (i=0;i<num;i++) imem[pim++] = rs[prst+i];
             imem[pim++] = MAGIC; imem[pim++] = code;
  }
}

void setpath (int status)
{
/* If BEGIN, sets the global pX, pY to current point. If CLOSE, restores
   current point to stored position
*/
  switch (status) {
    case BEGIN: inpath = 1;
                pX = cX; pY = cY;
                if (lX == MAGIC) {
                  lX = uX = cX;
                  lY = uY = cY;
                }
		break;
    case CLOSE: inpath = 0;
		break;
  }
}

void setbbox (int status)
{
/* moves the current point and, if path is currently drawn, setbbox()
   actualises the BoundingBox values
*/
  int dx, dy;
  switch (status) {
    case CURXY:  dx = rs[prst-2];  cX += dx;
                 dy = rs[prst-1];  cY += dy;
                 break;
    case CURX:   dx = rs[prst-1];  cX += dx;
                 break;
    case CURY:   dy = rs[prst-1];  cY += dy;
                 break;
    case CURXY6: dx = rs[prst-6];  cX += dx;
                 dy = rs[prst-5];  cY += dy;
                 dx = rs[prst-4];  cX += dx;
                 dy = rs[prst-3];  cY += dy;
                 dx = rs[prst-2];  cX += dx;
                 dy = rs[prst-1];  cY += dy;
                 break;
    case CURHV4: dx = rs[prst-4];  cX += dx;
                 dx = rs[prst-3];  cX += dx;
                 dy = rs[prst-2];  cY += dy;
                 dy = rs[prst-1];  cY += dy;
                 break;
    case CURVH4: dy = rs[prst-4];  cY += dy;
                 dx = rs[prst-3];  cX += dx;
                 dy = rs[prst-2];  cY += dy;
                 dx = rs[prst-1];  cX += dx;
                 break;
  }
  if (inpath) {
    lX = min (lX, cX);  lY = min (lY, cY);
    uX = max (uX, cX);  uY = max (uY, cY);
  }
}

void teststack(int num)
{
  if (prst<num) error ("stack underflow", NULL);
  return;
}

void read_code(int status)
{
/* reads code of SUB from config file and stores it to
   imem[] by save_code()
*/
  int i, code, addtail;
  float curXtmp, curYtmp;

  if (status)  /* status == 1 , SUB declaration */
    subr[numsubs].code = pim;   /* budeme ukladat kod procedury do imem */
  else         /* status == 0 , CHAR declaration */
    newchar[numchars].owncode = pim;
  scaleX = scaleY = 1;
  curX = curY = 0;
  vvax = lX = lY = uX = uY = MAGIC;
  inpath = 0;
  cX = cY = 0;
  prst = 0; addtail = 1;

  while ((code=read_token(fin, 1)) != EGROUP) {
    switch (code) {
      case EOF: error ("unexpected end of conf. file", NULL);
      case RETURN: addtail = 0; save_code(0, RETURN);
              break;
      case ENDCHAR: if (status) error("not allowed operator (%s)", buf);
              addtail = 0;  save_code(0, ENDCHAR);
              break;
      case UNKNOWN:
	      for (i=0; i<numsubs; i++) if (strcmp(buf,subr[i].name)==0) {
		  save_code(0, NUMSUB);
                  store_stack((float)i);
		  break;
              }
              if (i==numsubs) error ("unknown operator '%s' in SUB", buf);
      case VALUE: break; /* value is stored in rs[] by read_token() */
      case HSBW: save_code(2, HSBW);  break;
      case SBW:  save_code(4, SBW);   break;
      case SEAC: if (status) error("not allowed operator (%s)", buf);
                   save_code(5, SEAC);  break;
      case HSTEM: teststack(2);
                  rs[prst-2] *= scaleY;  rs[prst-1] *= scaleY;
                   save_code(2, code); break;
      case VSTEM: teststack(2);
                  rs[prst-2] *= scaleX;  rs[prst-1] *= scaleX;
                   save_code(2, code); break;
      case HSTEM3:  teststack(6);
                  for (i=0; i<6; i++) rs[prst-6+i] *= scaleY;
                   save_code(6, code); break;
      case VSTEM3: teststack(6);
                   for (i=0; i<6; i++) rs[prst-6+i] *= scaleX;
                   save_code(6, code); break;
      case DIV: save_code(MAXSTACK, DIV); /* save the previous number too */
                break;
      case CALLOTHERSUBR: save_code(MAXSTACK, CALLOTHERSUBR); break;
      case POP: save_code (0, POP); break;
      case SETCURRENTPOINT:  save_code(2, code); break;
      case MOVETO:
              teststack(2);
              rs[prst-2] -= curX;  rs[prst-1] -= curY;
      case RMOVETO:
              teststack(2);
              curX += rs[prst-2];  curY += rs[prst-1];
              rs[prst-2] *= scaleX;  rs[prst-1] *= scaleY;
              setbbox(CURXY);
              save_code(2, RMOVETO);
              break;
      case HMOVETO:
              teststack(1);
              curX += rs[prst-1];
              rs[prst-1] *= scaleX;
              setbbox(CURX);
              save_code(1, HMOVETO);
              break;
      case VMOVETO:
              teststack(1);
              curY += rs[prst-1];
              rs[prst-1] *= scaleY;
              setbbox(CURY);
              save_code(1, VMOVETO);
              break;
      case LINETO:  
              teststack(2);
              rs[prst-2] -= curX;  rs[prst-1] -= curY;
      case RLINETO:
              teststack(2);
              curX += rs[prst-2];  curY += rs[prst-1];
              rs[prst-2] *= scaleX;  rs[prst-1] *= scaleY;
              if (!inpath) setpath(BEGIN);
              setbbox(CURXY);
              save_code(2, RLINETO);
              break;
      case HLINETO:
              teststack(1);
              curX += rs[prst-1];
              rs[prst-1] *= scaleX;
              if (!inpath) setpath(BEGIN);
              setbbox(CURX);
              save_code(1, HLINETO);
              break;
      case VLINETO:
              teststack(1);
              curY += rs[prst-1];
              rs[prst-1] *= scaleY;
              if (!inpath) setpath(BEGIN);
              setbbox(CURY);
              save_code(1, VLINETO);
              break;
      case CURVETO:
              teststack(6);
              rs[prst-6] -= curX;  rs[prst-5] -= curY;
              curX += rs[prst-6];  curY += rs[prst-5];
              rs[prst-6] *= scaleX;  rs[prst-5] *= scaleY;
              rs[prst-4] -= curX;  rs[prst-3] -= curY;
              curX += rs[prst-4];  curY += rs[prst-3];
              rs[prst-4] *= scaleX;  rs[prst-3] *= scaleY;
              rs[prst-2] -= curX;  rs[prst-1] -= curY;
              curX += rs[prst-2];  curY += rs[prst-1];
              rs[prst-2] *= scaleX;  rs[prst-1] *= scaleY;
              if (!inpath) setpath(BEGIN);
              setbbox(CURXY6);
              save_code(6, RRCURVETO);
              break;
      case RCURVETO:
              teststack(6);
              curXtmp = curX;  curYtmp = curY;
              curX += rs[prst-6];  curY += rs[prst-5];
              rs[prst-6] *= scaleX;  rs[prst-5] *= scaleY;
              rs[prst-4] -= curXtmp - curX;  rs[prst-3] -= curYtmp - curY;
              curX += rs[prst-4];  curY += rs[prst-3];
              rs[prst-4] *= scaleX;  rs[prst-3] *= scaleY;
              rs[prst-2] -= curXtmp - curX;  rs[prst-1] -= curYtmp - curY;
              curX += rs[prst-2];  curY += rs[prst-1];
              rs[prst-2] *= scaleX;  rs[prst-1] *= scaleY;
              if (!inpath) setpath(BEGIN);
              setbbox(CURXY6);
              save_code(6, RRCURVETO);
              break;
      case RRCURVETO:
              teststack(6);
              curX += rs[prst-6];  curY += rs[prst-5];
              rs[prst-6] *= scaleX;  rs[prst-5] *= scaleY;
              curX += rs[prst-4];  curY += rs[prst-3];
              rs[prst-4] *= scaleX;  rs[prst-3] *= scaleY;
              curX += rs[prst-2];  curY += rs[prst-1];
              rs[prst-2] *= scaleX;  rs[prst-1] *= scaleY;
              if (!inpath) setpath(BEGIN);
              setbbox(CURXY6);
              save_code(6, RRCURVETO);
              break;
      case HVCURVETO:
              teststack(4);
              curX += rs[prst-4];
              rs[prst-4] *= scaleX;
              curX += rs[prst-3];  curY += rs[prst-2];
              rs[prst-3] *= scaleX;  rs[prst-2] *= scaleY;
              curY += rs[prst-1];
              rs[prst-1] *= scaleY;
              if (!inpath) setpath(BEGIN);
              setbbox(CURHV4);
              save_code(4, HVCURVETO);
              break;
      case VHCURVETO:
              teststack(4);
              curY += rs[prst-4];
              rs[prst-4] *= scaleY;
              curX += rs[prst-3];  curY += rs[prst-2];
              rs[prst-3] *= scaleX;  rs[prst-2] *= scaleY;
              curX += rs[prst-1];
              rs[prst-1] *= scaleX;
              if (!inpath) setpath(BEGIN);
              setbbox(CURVH4);
              save_code(4, VHCURVETO);
              break;
      case CALLSUBR:
	      save_code(1, CALLSUBR);
	      break;
      case SCALED:
              teststack(2);
              prst -= 2;
              if (prst<0) error ("stack underflow", NULL);
              scaleX = scaleX * rs[prst]; scaleY = scaleY * rs[prst+1];
	      break;
      case VVAXIS:
              teststack(1);
              prst -= 1;
              if (prst<0) error ("stack underflow", NULL);
              vvax = cX + scaleX * rs[prst];
              break;
      case CLOSEPATH: setpath(CLOSE);
      case DOTSECTION:
	      save_code(0, code);
              break;
      default: error("'%s' not allowed in SUB\n", buf);
    }
  }
  if (addtail)
    if (status) {
      if (inpath) save_code (0, CLOSEPATH);
      rs[prst++] = -cX;  rs[prst++] = -cY;
      save_code (2, RMOVETO);
      save_code (0, RETURN);
    } else {
      if (inpath) save_code (0, CLOSEPATH);
      save_code (0, ENDCHAR);
    }
  save_code (0, MAGIC);  /* the end signature */
  if (status) {
    if (lX == MAGIC) lX = lY = uX = uY = 0;
    subr[numsubs].lx = lX;
    subr[numsubs].ly = lY;
    subr[numsubs].ux = uX;
    subr[numsubs].uy = uY;
    if (vvax == MAGIC) subr[numsubs].vvaxis = (lX + uX) / 2 ;
    else subr[numsubs].vvaxis = vvax ;
    numsubs++;
  }
  return;
}

void print_code(unsigned int i)
{
/* print_code() prints the CharString or Subrs code from imem[]
   to pps output in readable form
*/
  unsigned j=0;
  char tabchar;

  tabchar = '\t';
  while (1) {
    if (imem[i]==MAGIC) {
      switch (imem[++i]) {
        case MAGIC: return;
        case HSTEM: j=0; break;
        case VSTEM: j=1; break;
        case VMOVETO: j=2; break;
        case RLINETO: j=3; break;
        case HLINETO: j=4; break;
        case VLINETO: j=5; break;
        case RRCURVETO: j=6; break;
        case CLOSEPATH: j=7; break;
        case CALLSUBR: j=8; break;
        case RETURN: j=9; break;
        case ESCAPE: j=10; break;
        case HSBW: j=11; break;
        case ENDCHAR: j=12; break;
        case RMOVETO: j=13; break;
        case HMOVETO: j=14; break;
        case VHCURVETO: j=15; break;
        case HVCURVETO: j=16; break;
        case DOTSECTION: j=17; break;
        case VSTEM3: j=18; break;
        case HSTEM3: j=19; break;
        case SEAC: j=20; break;
        case SBW: j=21; break;
        case DIV: j=22; break;
        case CALLOTHERSUBR: j=23; break;
        case POP: j=24; break;
        case SETCURRENTPOINT: j=25; break;
        case NUMSUB: j=30; break;
      }
      if (j==30) {
        fprintf(fpps, "%c%d", tabchar, imem[++i]+pfbnumsubs);
        tabchar = ' ';
      } else {
        j += start_type1code;
        fprintf(fpps, "%c%s\n", tabchar, csname[j]);
        tabchar = '\t';
      }
    }
    else {
      fprintf(fpps, "%c%d", tabchar, imem[i]);
      tabchar = ' ';
    }
    i++;
  }
}


void print_encoding(void)
{
/* prints the encoding vector to pps file
*/
  int code;
  for(code=0;code<256;code++)
    if (encvector[code]!=NULL)
      fprintf(fpps, "dup %d /%s put\n", code, encvector[code]);
}

void read_in(void)
{
/* reads an IN list of input files
*/
  while (1) {
    if(read_token(fin,0)==EOF) return;
    if (strcmp(buf,"END")==0 ||
        strcmp(buf,"ENC")==0 ||
        strcmp(buf,"REM")==0 ||
        strcmp(buf,"SRC")==0 ||
        strcmp(buf,"NAME")==0 ||
        strcmp(buf,"VAR")==0 ||
        strcmp(buf,"SUB")==0 ||
        strcmp(buf,"CHAR")==0)  return;
    if (strcmp(buf,"IN")==0) { read_in(); return; }
    if (numfiles>MAXINFILES-2)
       error ("MAXINFILES (max number of IN files) overflow", NULL);
    infile[numfiles++] = add_gbuf(buf);
  }
}


void read_enc(void)
{
/* reads an ENC declaration plus .enc file
*/
  int c, p, code, saveline;

  if(encisread) error("second ENC declaration is not allowed", NULL);
  if(read_token(fin,0)==EOF) error("unexcepted end of file", NULL);
  if(strlen(buf) > MAXBUF-5) error("ENC file name is too long (%s)", buf);
  strcat(buf, ".enc");  strcpy(fname, buf);

  fenc = fopen(fname, "r");
  if (!fenc) error("cannot open %s for reading\n", fname);
  saveline = line; line = 1;
  c = 0;
  while (c!='[' && c!=EOF) c = Fgetc(fenc);
  if (c==EOF) error ("no '[' in encoding file %s", fname);
  code = 0;
  while (1) {
    while (c!='/' && c!=']' && c!=EOF) c = Fgetc(fenc);
    if (c==EOF) error ("unexpected end of encoding file %s", fname);
    if (c==']' && code==256) break;
    if (c==']') error ("less than 256 entries in encoding vector (/%s)",buf);
    p=0;
    c = Fgetc(fenc);
    while (c!=' ' && c!='/' && c!=EOF) {
      if (p>MAXBUF-2) error ("too long char name (%s) in enc file", buf);
      buf[p++] = c;
      c = Fgetc(fenc);
    }
    if (p==0) error ("empty char name in enc. file %s", fname);
    if (code > 255)
      error ("more than 256 entries in encoding vector (/%s)",buf);
    buf[p] = 0;
    if (strcmp(buf,".notdef")==0) encvector[code++] = NULL;
    else encvector[code++] = add_gbuf(buf);
  }
  fclose(fenc);
  line = saveline;
  encisread=1;
}

void read_rem(void)
{
/* reads a remark string
*/
  int c;

  c = ' ';
  while (c==' ') c = Fgetc(fin);
  if (c != '"') error ("the '\"' (begin of string) after REM expected", NULL);
  if (numrems >= MAXREMARKS)
    error ("MAXREMARKS (maximum REM lines) exceeded", NULL);
  remark[numrems++] = pbuf;
  while ((c=Fgetc(fin)) != '"' && c != EOF && pbuf < &gbuf[MAXGBUF]) {
    pbuf[0] = c;
    pbuf++;
  }
  if (c == EOF) error ("the REM string is unterminated", NULL);
  if (pbuf == &gbuf[MAXGBUF])
    error ("MAXGBUF overflow during REM reading", NULL);
  store_gbuf(0);
}

void read_var(void)
{
/* reads the VAR declaration from config file
*/
  int c;
  c = ' ';
  while (c==' ') c = Fgetc(fin);
  if (c!='/') error("the /name of variable expected", NULL);
  read_token(fin, 0);
  if (buf[0]==0) error("empty /name of variable", NULL);
  variable[numvars].name = add_gbuf(buf);
  if (read_token(fin,1) != VALUE)  error("the number expected (%s)", buf);
  variable[numvars++].value = rs[--prst];
  return;
}

void read_src(void)
{
  if(srcisread) error("second SRC declaration is not allowed", NULL);
  read_token(fin, 0);
  srcdir = add_gbuf(buf);
  srcisread = 1;
}

void read_name(void)
{
  if(nameisread) error("second NAME declaration is not allowed", NULL);
  read_token(fin, 0);
  oriname = add_gbuf(buf);
  read_token(fin, 0);
  if (strcmp (buf, "->") != 0) error("the '->' expected", NULL);
  read_token(fin, 0);
  newname = add_gbuf(buf);
  diffname = strlen(newname) - strlen(oriname);
  nameisread = 1;
}

void read_bbox(void)
{
  int i=0;
  if (read_token(fin,1) != VALUE)  i = 1;
  if (read_token(fin,1) != VALUE)  i = 1;
  if (read_token(fin,1) != VALUE)  i = 1;
  if (read_token(fin,1) != VALUE)  i = 1;
  if (i) error("the number expected (%s)", buf);
  glX = min(glX, rs[prst-4]);
  glY = min(glY, rs[prst-3]);
  guX = max(guX, rs[prst-2]);
  guY = max(guY, rs[prst-1]);
  prst = 0;
}

void read_sub(void)
{
/* reads the SUB declaration from config file
*/
  int c;
  c = ' ';
  while (c==' ') c = Fgetc(fin);
  if (c!='/') error("the /name of subroutine expected", NULL);
  read_token(fin, 0);
  if (buf[0]==0) error("empty /name of subroutine", NULL);
  if (numsubs>=MAXSUBS) error ("MAXSUBS (number of SUB's) exceeded", NULL);
  subr[numsubs].name = add_gbuf(buf);
  c = ' ';
  while (c==' ') c = Fgetc(fin);
  if (c!='{') error("the '{' expected", NULL);
  read_code(1);
  return;
}

void read_char(void)
{
/* reads the CHAR declaration from config file
*/
  int c, p;
  c = ' ';
  while (c==' ') c = Fgetc(fin);
  if (c!='/') error("the /name of new char expected", NULL);
  read_token(fin, 0);
  if (buf[0]==0) error("empty /name new char", NULL);
  newchar[numchars].name = add_gbuf(buf);
  c = ' ';
  while (c==' ') c = Fgetc(fin);
  if (c=='{') {
    read_code(0);
    newchar[numchars].nsub = -1;
    newchar[numchars++].origname = NULL;
    return;
  }
  if (c!='/') error("the /name of base char or '{' expected", NULL);
  read_token(fin, 0);
  if (buf[0]==0) error("empty /name of base char", NULL);
  newchar[numchars].origname = add_gbuf(buf);
  newchar[numchars].owncode = 0;
  c = ' ';
  while (c==' ') c = Fgetc(fin);
  if (c!='/') {
    ungetc(c, fin);
    if (read_token(fin, 1) != VALUE)
       error("value or /name of subroutine expected (%s)", buf);
    if (read_token(fin, 1) != VALUE) error("value expected (%s)", buf);
    if (read_token(fin, 1) != VALUE) error("value expected (%s)", buf);
    prst -= 3;
    newchar[numchars].dsbx = rs[prst];
    newchar[numchars].dy = rs[prst+1];
    newchar[numchars].dwx = rs[prst+2];
    read_token(fin, 0);
    if (strcmp(buf,"correct")!=0)
       error("the correct operator expected (%s)", buf);
    c = ' ';
    while (c==' ') c = Fgetc(fin);
  } else
    newchar[numchars].dsbx = newchar[numchars].dwx = newchar[numchars].dy = 0;
  if (c!='/') error("the /name of subroutine expected (%s)", buf);
  read_token(fin, 0);
  if (buf[0]==0) error("empty /name of subroutine", NULL);
  newchar[numchars].nsub = -1;
  if (strcmp(buf,".noinsert")==0) { numchars++;  return; }
  for(p=0;p<numsubs;p++) if (strcmp(buf,subr[p].name)==0) {
    newchar[numchars].nsub = p;
    break;
  }
  if (newchar[numchars].nsub == -1)
    error("unknown name of subroutine /%s", buf);
  if (read_token(fin, 1) != VALUE) error("value expected (%s)", buf);
  if (read_token(fin, 1) != VALUE) error("value expected (%s)", buf);
  prst -= 2;
  newchar[numchars].x = rs[prst]; newchar[numchars].y = rs[prst+1];
  read_token(fin, 0);
  p = -1;
  if (strcmp(buf,"putorigin")==0)  p = PUTORIGIN;
  if (strcmp(buf,"putsidebar")==0) p = PUTSIDEBAR;
  if (strcmp(buf,"putcenter")==0)  p = PUTCENTER;
  if (strcmp(buf,"putafter")==0)   p = PUTAFTER;
  if (p == -1) error("unknown put- operator (%s)", buf);
  newchar[numchars++].type = p;
  return;
}

void read_config(void)
{
/* reads the whole config file
*/
  while (1) {
    if(read_token(fin,0)==EOF) return;
    if (strcmp(buf,"IN")==0) read_in();
    if (strcmp(buf,"END")==0) return;
    if (strcmp(buf,"ENC")==0) read_enc();
    if (strcmp(buf,"REM")==0) read_rem();
    if (strcmp(buf,"VAR")==0) read_var();
    if (strcmp(buf,"SRC")==0) read_src();
    if (strcmp(buf,"NAME")==0) read_name();
    if (strcmp(buf,"SUB")==0) read_sub();
    if (strcmp(buf,"CHAR")==0) read_char();
    if (strcmp(buf,"BBOX")==0) read_bbox();
  }
}

void globalbbox(void)
{
/* computes the global glY and guY of BoundingBox.
*/
  int i;
  for (i=0; i<numchars; i++) if (newchar[i].nsub>=0) {
    glY = min(glY, subr[newchar[i].nsub].ly + newchar[i].y + newchar[i].dy);
    guY = max(guY, subr[newchar[i].nsub].uy + newchar[i].y + newchar[i].dy);
  }
}

unsigned int convert_code(unsigned char* st, int l)
{
/* converts the decrypted CharString or Subrs from
   per-byte format to imem[] format.
*/
  int p;
  int32 val;
  int b;
  unsigned int out;

  out = pim;
  for (p=0; p<l; p++) {
    b = st[p];
    if (b >= 32) {
      if (b >= 32 && b <= 246) val = b - 139;
      else if (b >= 247 && b <= 250) val = (b - 247)*256 + 108 + st[++p];
      else if (b >= 251 && b <= 254) val = -(b - 251)*256 - 108 - st[++p];
      else {
        val =  (st[++p] & 0xff) << 24;
        val |= (st[++p] & 0xff) << 16;
        val |= (st[++p] & 0xff) <<  8;
        val |= (st[++p] & 0xff) <<  0;
        /* in case an int32 is larger than four bytes---sign extend */
        #if MAX_INT > 0x7FFFFFFFUL
          for (i = 4; i < sizeof(int32); i++)
            val |= 0xff << (i * 8);
        #endif
      }
      imem[pim++] = val;
    } else {
      imem[pim++] = MAGIC;
      if (b==12) b = 100*b + st[++p];
      imem[pim++] = b;
    }
    if (pim >= MAXIMEM-5) error ("imem[MAXIMEM] overflow", NULL);
  }
  imem[pim++] = MAGIC; imem[pim++] = MAGIC;
  topim = pim;
  return out;
}

void insert_newcode (unsigned int pim, int p)
{
/* insert a new code:  dx dy rmoveto num callsubr -dx -dy rmoveto
   into code at imem[i] and does some optimalisation 
*/
  unsigned int npim, lpim, i;
  int dx, dy, wx, sbx, c, shift;
  int tmem[] = {0, 0, MAGIC, RMOVETO, 
                0, MAGIC, CALLSUBR, 
                0, 0, MAGIC, RMOVETO};

  lpim = 11;
  wx = MAGIC; sbx = 0;
  while (1) {
    c = imem[pim++];
    if (c != MAGIC) continue;
    c = imem[pim++];
    if (c==VSTEM || c==VSTEM3 || c==DIV)  continue;
    if (c==HSTEM) {
      imem[pim-4] += newchar[p].dy;
      continue;
    }
    if (c==HSTEM3) {
      imem[pim-8] += newchar[p].dy;
      imem[pim-6] += newchar[p].dy;
      imem[pim-4] += newchar[p].dy;
      continue;
    }
    if (c==HSBW) {
      if (imem[pim-4]==MAGIC && imem[pim-3]==DIV) {
        imem[pim-6] += newchar[p].dwx * imem[pim-5];
        wx = imem[pim-6] / imem[pim-5];
        imem[pim-7] += newchar[p].dsbx;
        sbx = imem[pim-7];
      } else {
        imem[pim-3] += newchar[p].dwx;    wx = imem[pim-3];
        imem[pim-4] += newchar[p].dsbx;   sbx = imem[pim-4];
      }
      continue;
    }
    if (c==SEAC || c==SBW) {
        fprintf
           (stderr,"\nWarning: seac or sbw in base char, /%s unchanged\n",
           newchar[p].name);
        return;
    }
    if (c==MAGIC) {
      fprintf(stderr, "\nWarning: the /%s is unchanged\n", newchar[p].name);
      return;
    }
    if (wx==MAGIC) {
      fprintf
         (stderr,"\nWarning: hsbw not found in base char, /%s unchanged\n",
         newchar[p].name);
      return;
    }
    if (newchar[p].nsub == -1) {
      if (newchar[p].dy == 0) return;
      tmem[0] = 0;  tmem[1] = newchar[p].dy;
      tmem[2] = MAGIC; tmem[3] = RMOVETO;
      lpim = 4;
    } else {
      dx = newchar[p].x; dy = newchar[p].y + newchar[p].dy;
      if (newchar[p].type == PUTORIGIN)  dx -= sbx;
      if (newchar[p].type == PUTCENTER)
        dx += wx/2 - sbx - subr[newchar[p].nsub].vvaxis;
      if (newchar[p].type == PUTAFTER)  dx += wx-sbx;

      if (dx == 0 && dy == 0) { /* remove rmoveto */
        tmem[0] = newchar[p].nsub + pfbnumsubs;
        tmem[1] = MAGIC;  tmem[2] = CALLSUBR;
        tmem[3] = 0;  tmem[4] = newchar[p].dy;
        tmem[5] = MAGIC;  tmem[6] = RMOVETO;
        lpim = 7;
      }
      else if (dx == 0) {  /* rmoveto -> vmoveto */
        tmem[0] = dy;
        tmem[1] = MAGIC; tmem[2] = VMOVETO;
        tmem[3] = newchar[p].nsub + pfbnumsubs;
        tmem[4] = MAGIC;  tmem[5] = CALLSUBR;
        tmem[6] = 0; tmem[7] = -dy + newchar[p].dy;
        tmem[8] = MAGIC;  tmem[9] = RMOVETO;
        lpim = 10;
      }
      else if (dy == 0) {  /* rmoveto -> hmoveto */
        tmem[0] = dx;
        tmem[1] = MAGIC; tmem[2] = HMOVETO;
        tmem[3] = newchar[p].nsub + pfbnumsubs;
        tmem[4] = MAGIC;  tmem[5] = CALLSUBR;
        tmem[6] = -dx; tmem[7] = newchar[p].dy;
        tmem[8] = MAGIC;  tmem[9] = RMOVETO;
        lpim = 10;
      } else {              /* default */
        tmem[0] = dx;  tmem[1] = dy;
        tmem[4] = newchar[p].nsub + pfbnumsubs;
        tmem[7] = -dx; tmem[8] = -dy + newchar[p].dy;
      }
    }
    if (c==RMOVETO) {      /* next command is rmoveto */
      tmem[lpim-4] += imem[pim-4] ;
      tmem[lpim-3] += imem[pim-3] ;
      npim = pim - 4;
    }
    else if (c==VMOVETO) {  /* next command is vmoveto */
      tmem[lpim-3] += imem[pim-3] ;
      npim = pim - 3;
    }
    else if (c==HMOVETO) {  /* next command is hmoveto */
      tmem[lpim-4] += imem[pim-3];
      npim = pim - 3;
    }
    else {                  /* next command is another */
      pim -= 3;
      while (imem[pim] != MAGIC && pim >= fixpim) pim--;
      if (imem[pim] != MAGIC) error ("something wrong", NULL);
      pim += 2;
      npim = pim;
    }
    dx = tmem[lpim-4];  dy = tmem[lpim-3];
    if (dx == 0 && dy == 0) lpim -=4 ;  /* new rmoveto is irrelevant */
    else if (dx == 0) {                 /* new vmoveto */
      tmem[lpim-4] = dy;
      tmem[lpim-3] = MAGIC;  tmem[lpim-2] = VMOVETO;
      lpim--;
    }
    else if (dy == 0) {                  /* new hmoveto */
      tmem[lpim-3] = MAGIC;  tmem[lpim-2] = HMOVETO;
      lpim--;
    }
    shift = lpim - (pim - npim);
    if (topim+shift >= MAXIMEM) error ("MAXIMEM overflow", NULL);
    if (shift < 0) for (i=pim; i<topim; i++) imem[i+shift] = imem[i];
    if (shift > 0) for (i=topim-1; i>=pim; i--) imem[i+shift] = imem[i];
    topim += shift;
    for (i=0; i<lpim; i++) imem[npim+i] = tmem[i];
    return;
  }
}

void print_subrs(void)
{
/* prints all /Subrs (original plus new) to pps
*/
  int p;
  for (p=0;p<pfbnumsubs;p++) {
    fprintf (fpps, "dup  %d {\n", p);
    fixpim = convert_code(savedsubr[p].code, savedsubr[p].len);
    print_code (fixpim);
    fprintf (fpps, "\t} %s\n", NPcommand);
    pim = fixpim;
  }
  for (p=0;p<numsubs;p++) {
    fprintf (fpps, "dup %d {\n", p + pfbnumsubs);
    print_code (subr[p].code);
    fprintf (fpps, "\t} %s\n", NPcommand);
  }
}

void print_chars(void)
{
/* prints all /CharStrings (original plus new) to pps
*/
  int p;
  
  for (p=0; p<pfbchars; p++) if (savedchars[p].name) {
    fprintf (fpps, "/%s {\n", savedchars[p].name);
    fixpim = convert_code(savedchars[p].code, savedchars[p].len);
    print_code (fixpim);
    fprintf (fpps, "\t} %s\n", NDcommand);
    pim = fixpim;
  }
  for (p=0; p<numchars; p++)
  if (!newchar[p].origname || newchar[p].origcode) {

    fprintf (fpps, "/%s {\n", newchar[p].name);
    if (newchar[p].origname == NULL)
      print_code (newchar[p].owncode);
    else {
      fixpim = convert_code(newchar[p].origcode, newchar[p].origlen);
      insert_newcode (fixpim, p);
      print_code (fixpim);
      pim = fixpim;
    }
    fprintf (fpps, "\t} %s\n", NDcommand);
  }
}

void check_doublechars()
{
/* checks the chars from pfbchars \cap newchars.
   If not empty, does the warning. Then check_doublechars() sets the
   origcode to newchar array and calculates the sumchars ... the number
   of pfbchars \cup newchars
*/
  int i, j, empty;

  empty = 1;

  for (i=0; i<pfbchars; i++) 
    for (j=0; j<numchars; j++)
      if (strcmp(savedchars[i].name,newchar[j].name)==0) {
	empty = 0;
        savedchars[i].code = NULL;
	break;
      }
  if (!empty) {  /* print warning */
    fprintf(stderr, "\nWarning: the following CharNames will be replaced");
    fprintf(stderr, " by new definition:\n");
    for (i=0; i<pfbchars; i++)
      if (savedchars[i].code == NULL) {
	 fprintf(stderr, " /%s", savedchars[i].name);
	 savedchars[i].name = NULL;
      }
    fprintf(stderr, "\n");
  }
  empty = 1;
  for (j=0; j<numchars; j++) {
    newchar[j].origcode = NULL;
    for (i=0; i<pfbchars; i++)
      if (newchar[j].origname &&
          strcmp(newchar[j].origname,savedchars[i].name)==0) {
              newchar[j].origcode = savedchars[i].code;
              newchar[j].origlen  = savedchars[i].len;
              break;
      }
    if (newchar[j].origcode == NULL && newchar[j].origname != NULL) empty = 0;
  }
  if (!empty) { /* print warning */
    fprintf(stderr, "\nWarning: the following CharNames will be ignored");
    fprintf(stderr, " because of no basechar exists:\n");
    for (j=0; j<numchars; j++)
      if (newchar[j].origcode == NULL && newchar[j].origname != NULL)
	 fprintf(stderr, " /%s", newchar[j].name);
    fprintf(stderr, "\n");
  }
  sumchars = 0;
  for (i=0; i<pfbchars; i++)
    if (savedchars[i].name != NULL) sumchars++;
  for (j=0; j<numchars; j++)
    if (newchar[j].origcode != NULL || newchar[j].origname == NULL) sumchars++;
  return;
}

int getpfb(void)
{
/* getpfb() gets new byte from pfb, decrypts it and check the pattern
*/
  int c;

  c = egetc();
  if (c==EOF) error ("text '%s' not found", pattern);
  if (c==pattern[pip]) {
    pip++;
    if (pattern[pip]==0) ispattern = 1;
  }
  else pip = 0;
  return c;
}

void set_pattern(char *pat)
{
/* new pattern is set for checking the input from pfb
*/
  ispattern = 0; pip = 0;
  strcpy (pattern, pat);
}

void save_string(num, status)
{
/* save_string stores the decrypted per-byte CharString or Subr
   from pfb to global buffer
*/
  int c, p, l;

  set_pattern(RDcommand);
  p = 0;
  while(!ispattern) buf[p++] = getpfb();
  p -= 4;
  while(p>0 && buf[p] != ' ') p--;
  if (p==0) error ("the %s parameter not found", RDcommand);
  l = atoi (&buf[p]);
  l -= lenIV;
  getpfb();  /* ignore the space after RDcommand */

  if (status==SAVECHAR) {
    while(p>=0 && buf[p] == ' ') p--;
    if (p<0) error ("the CharName not found", NULL);
    buf[p+1] = 0;
    savedchars[num].name = add_gbuf(buf);
    savedchars[num].code = pbuf;
    savedchars[num].len = l;
  } else {
    savedsubr[num].len = l;
    savedsubr[num].code = pbuf;
  }
  is_textspace = 0;
  cr = 4330;
  for (p=0; p<lenIV; p++) (void) cgetc();
  for (p=0; p<l; p++) {
    c = cgetc();
    store_gbuf(c);
  }
  is_textspace = 1;
}

void find_text (char *text, char *command)
{
/* searchs the text in gbuf and save the /string preceeded this text
   to command.
*/
  unsigned char *found;
  int p;

  found = strstr(pbuf, text);
  if (!found) error ("string '%s' not found", text);
  while (found > pbuf && found[0]!='/') found--;
  if (found==pbuf) error ("'/?? {%s...' not found",text);
  found++;
  p = 0;
  while (p<3 && found[p]!=' ' && found[p]!='{') command[p] = found[p++];
  if (p==3) error ("'/?? {%s...' must be two character string", text);
  command[p] = 0;
}


void replace_name (void)
{
  unsigned char *lbuf;
  int i, len;

  if (!nameisread) return;
  lbuf = strstr (pbuf, oriname);
  while (lbuf) {
    if (diffname > 0) {
      i = strlen(lbuf);
      while (i>0) lbuf[i+diffname] = lbuf[i--];
    }
    if (diffname < 0) {
      i = -diffname;
      while (lbuf[i]) lbuf[i+diffname] = lbuf[i++];
      lbuf[i+diffname] = 0;
    }
    len = strlen(newname);
    for (i=0; i<len; i++)  lbuf[i] = newname[i];
    lbuf = &lbuf[len];
    lbuf = strstr (lbuf, oriname);
  }
}

void convert_pfb(void)
{
/* does the main work. Converts the pfb input to pps output.
*/
  char *found;
  char dupbuf[4];
  int c, p, i, bb[4];

  immediate_eexec = 0; in_eexec = 0;
  first_byte = 1;
  pfb_remaining = 0;
  savedchar = -1;
  is_textspace = 1;
  fixpbuf = pbuf;
  lenIV = 4;

  set_pattern("\n");
  while (!ispattern) store_gbuf (getpfb());
  pbuf[0] = 0;  pbuf = fixpbuf;
  replace_name ();
  fputs (pbuf, fpps);

  for (i=0; i<numrems; i++) fprintf (fpps, "%% %s\n", remark[i]);

  set_pattern("/Encoding");
  while (!ispattern) store_gbuf (getpfb());
  pbuf[0] = 0; pbuf = fixpbuf;
  replace_name ();
  fputs (pbuf, fpps);

  set_pattern(" def");
  while (!ispattern) getpfb();
  fprintf (fpps, " 256 array\n0 1 255 {1 index exch /.notdef put} for\n");
  print_encoding();
  fprintf (fpps, "readonly def");

  set_pattern("/FontBBox");
  while (!ispattern) fputc(getpfb(), fpps);
  c = ' ';  p = 0;
  while (c!=']' && c!='}' && c!='\n') c = buf[p++] = getpfb();
  buf[p] = 0; p = 0;
  for (i=0; i<4; i++) {
    while (buf[p] && !isdigit(buf[p]) && buf[p]!='-') p++;
    if (buf[p]==0) error("cannot able to read FontBBox (%s)", buf);
    bb[i] = atoi(&buf[p]);
    while (buf[p] && buf[p]!=' ') p++;
  }
  fprintf (fpps, " [%d %d %d %d]",
    min(bb[0],glX), min(bb[1],glY), max(bb[2],guX), max(bb[3],guY));

  set_pattern("currentfile eexec");
  while (!ispattern) fputc(getpfb(), fpps);
  immediate_eexec = 1;

  set_pattern("/Subrs");
  while (!ispattern) {
    c = getpfb();
    store_gbuf(c);
    fputc(c, fpps);
  }
  pbuf[0] = 0;  pbuf = fixpbuf;

  find_text ("string currentfile", RDcommand);
  find_text ("noaccess def", NDcommand);
  find_text ("noaccess put", NPcommand);
  found = strstr(pbuf, "/lenIV ");
  if (found && isdigit(found[7]))  lenIV = atoi(found + 7);

  pfbnumsubs = 0;
  fixpim = pim;
  while (1) {
    set_pattern("\n");
    while (!ispattern) getpfb();
    for (p=0;p<3;p++) dupbuf[p] = getpfb();
    dupbuf[3] = 0;
    if (strcmp(dupbuf,"dup") != 0) break;
    save_string (pfbnumsubs++, SAVESUBR);
  }
  p = pfbnumsubs + numsubs;
  fprintf(fpps, " %d array\n", p);
  print_subrs();    /* prints orig + new subrs in readable form */
  fputs (dupbuf, fpps);
  pbuf = fixpbuf;
  pim = fixpim;     /* restore gbuf and imem pointers */

  set_pattern("/CharStrings");
  while (!ispattern) fputc(getpfb(), fpps);

  pfbchars = 0;
  while (1) {
    set_pattern("\n");
    while (!ispattern) getpfb();
    c = getpfb();
    if (c != '/') break;
    save_string (pfbchars++, SAVECHAR);
  }
  check_doublechars();
  fprintf(fpps, " %d dict dup begin\n", sumchars);
  print_chars();   /* prints orig + new CharScrings in readable form */
  fputc(c, fpps);

  set_pattern("currentfile closefile");
  while (!ispattern) fputc(getpfb(), fpps);
  
  fprintf (fpps, "\n");
  pbuf = fixpbuf;
  pim = fixpim;
}

int main(int argc, char **argv)
{
  int p;
  fprintf(stderr, "This is t1accent, (c) Petr Olsak, Apr. 1998\n");
  if (argc < 2) { 
    fprintf (stderr, "   usage: t1accent config.file.\n"
                     "   see t1accent.doc for more details.\n");
    return 1;
  }
  fin = fopen(argv[1], "r");
  if (!fin) error("cannot open config file %s for reading", argv[1]);

  init_csnames();
  line = 1;
  glX = glY = 1000;  guX = guY = -1000;

  read_config();
  globalbbox();
  line = 0;
  if (!encisread) error("no ENC declaration in config file %s", argv[1]);
  for(p=0;p<numfiles;p++) {
    strcpy(fname, infile[p]);
    if(strlen(fname) > MAXBUF-5) error("pfb file name is too long (%s)",fname);
    strcat(fname, ".pfb");
    if (srcisread) {
      if (strlen(srcdir)+strlen(fname) >= MAXBUF)
        error("dirname+pfb file name is too long (%s)",srcdir);
      strcpy (buf, fname);
      strcpy (fname, srcdir);
      strcat (fname, buf);
    }
    fpfb = fopen(fname, "rb");
    if (!fpfb) error("cannot open pfb file %s", fname);
    fprintf (stderr, "(%s", fname); 
    strcpy (buf, infile[p]); strcat(buf,".pps");
    fpps = fopen(buf, "w");
    if (!fpps) error("cannot write to file %s", buf);
    convert_pfb();
    fclose(fpfb);
    fclose(fpps);
    fprintf (stderr, "->%s.pps)\n", infile[p]);
  }
  return 0;
}

